//                                               -*- C++ -*-
/**
 *  @file  OrthogonalProductPolynomialFactory.cxx
 *  @brief This is the orthogonal polynomial basis
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "OrthogonalProductPolynomialFactory.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"
#include "Exception.hxx"
#include "Indices.hxx"
#include "ComposedDistribution.hxx"
#include "NumericalMathFunctionImplementation.hxx"
#include "ProductPolynomialEvaluationImplementation.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {
      using Uncertainty::Algorithm::OrthogonalUniVariatePolynomialFamily;

      TEMPLATE_CLASSNAMEINIT(PersistentCollection<OrthogonalUniVariatePolynomialFamily>);

      static Common::Factory<PersistentCollection<OrthogonalUniVariatePolynomialFamily> > RegisteredFactory("PersistentCollection<OrthogonalUniVariatePolynomialFamily>");

    } /* namespace Type */
  } /* namespace Base */

  namespace Uncertainty
  {

    namespace Algorithm
    {

      CLASSNAMEINIT(OrthogonalProductPolynomialFactory);

      static Base::Common::Factory<OrthogonalProductPolynomialFactory> RegisteredFactory("OrthogonalProductPolynomialFactory");

      typedef Base::Common::InvalidArgumentException                          InvalidArgumentException;
      typedef Base::Type::Indices                                             Indices;
      typedef Distribution::ComposedDistribution                              ComposedDistribution;
      typedef ComposedDistribution::DistributionCollection                    DistributionCollection;
      typedef Base::Func::NumericalMathFunctionImplementation                 NumericalMathFunctionImplementation;
      typedef Base::Func::ProductPolynomialEvaluationImplementation           ProductPolynomialEvaluationImplementation;
      typedef ProductPolynomialEvaluationImplementation::PolynomialCollection PolynomialCollection;

      /* Default constructor */
      OrthogonalProductPolynomialFactory::OrthogonalProductPolynomialFactory()
	: OrthogonalFunctionFactory(),
	  coll_(),
	  phi_()
      {
	// Nothing to do
      }


      /* Constructor */
      OrthogonalProductPolynomialFactory::OrthogonalProductPolynomialFactory(const PolynomialFamilyCollection & coll)
	: OrthogonalFunctionFactory(),
	  coll_(coll),
	  phi_(coll.getSize())
      {
	buildMeasure();
      }


      /* Constructor */
      OrthogonalProductPolynomialFactory::OrthogonalProductPolynomialFactory(const PolynomialFamilyCollection & coll,
									     const EnumerateFunction & phi)
	: OrthogonalFunctionFactory(),
	  coll_(coll),
	  phi_(phi)
      {
	if (coll.getSize() != phi.getDimension()) throw InvalidArgumentException(HERE) << "Error: the enumerate function must have a dimension equal to the collection size";
	buildMeasure();
      }


      /* Virtual constructor */
      OrthogonalProductPolynomialFactory * OrthogonalProductPolynomialFactory::clone() const
      {
	return new OrthogonalProductPolynomialFactory(*this);
      }


      /* Return the enumerate function that translate unidimensional indices nto multidimensional indices */
      EnumerateFunction OrthogonalProductPolynomialFactory::getEnumerateFunction() const
      {
	return phi_;
      }


      /* Build the NumericalMathFunction of the given index */
      OrthogonalProductPolynomialFactory::NumericalMathFunction OrthogonalProductPolynomialFactory::build(const UnsignedLong index) const
      {
	// Compute the multi-indices using the EnumerateFunction
	Indices indices(phi_(index));
	const UnsignedLong size(indices.getSize());
	// Then build the collection of polynoms using the collection of factories
	PolynomialCollection polynoms(size);
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    polynoms[i] = coll_[i].build(indices[i]);
	  }
	return NumericalMathFunctionImplementation(ProductPolynomialEvaluationImplementation(polynoms).clone());
      }


      /* String converter */
      String OrthogonalProductPolynomialFactory::__repr__() const
      {
	return OSS() << "class=" << getClassName()
		     << " univariate polynomial collection=" << coll_
		     << " measure=" << measure_;
      }

      
      /* Method save() stores the object through the StorageManager */
      void OrthogonalProductPolynomialFactory::save(const StorageManager::Advocate & adv) const
      {
	OrthogonalFunctionFactory::save(adv);
	adv.writeValue(coll_, StorageManager::MemberNameAttribute, "coll_");
      }


      /* Method load() reloads the object from the StorageManager */
      void OrthogonalProductPolynomialFactory::load(const StorageManager::Advocate & adv)
      {
	OrthogonalFunctionFactory::load(adv);
	adv.readValue(coll_, StorageManager::MemberNameAttribute, "coll_");
      }

      /* Build the measure based on the one found in the family collection */
      void OrthogonalProductPolynomialFactory::buildMeasure()
      {
	const UnsignedLong size(coll_.getSize());
	DistributionCollection distributions(size);
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    distributions[i] = coll_[i].getMeasure();
	  }
	measure_ = ComposedDistribution(distributions);
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
