//                                               -*- C++ -*-
/**
 *  @file  HermiteFactory.cxx
 *  @brief Hermite polynomial factory
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "HermiteFactory.hxx"
#include "PersistentObjectFactory.hxx"
#include "Normal.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      CLASSNAMEINIT(HermiteFactory);

      static Base::Common::Factory<HermiteFactory> RegisteredFactory("HermiteFactory");

      typedef Distribution::Normal                   Normal;
      typedef Base::Common::InvalidArgumentException InvalidArgumentException;

      /* Default constructor */
      HermiteFactory::HermiteFactory()
	: OrthogonalUniVariatePolynomialFactory(Normal())
      {
	initializeCache();
      }


      /* Virtual constructor */
      HermiteFactory * HermiteFactory::clone() const
      {
	return new HermiteFactory(*this);
      }


      /* Determine the coefficients of the first Orthonormal Hermite polynomial*/
      HermiteFactory::Coefficients HermiteFactory::getP0Coefficients() const
      {
	Coefficients Hermite0Coefficients(1); 
	Hermite0Coefficients[0] = 1.0;
	return Hermite0Coefficients;
      }


      /* Determine the coefficients of the second Orthonormal Hermite polynomial*/
      HermiteFactory::Coefficients HermiteFactory::getP1Coefficients() const
      {
	Coefficients Hermite1Coefficients(2); 
	Hermite1Coefficients[0] = 0.0; 
	Hermite1Coefficients[1] = 1.0; 
	return Hermite1Coefficients;
      }


      /* Calculate the coefficients of recurrence a0, a1, a2 such that
	 Pn(x) = (a0 * x + a1) * Pn-1(x) + a2 * Pn-2(x) */
      HermiteFactory::Coefficients HermiteFactory::getRecurrenceCoefficients(const UnsignedLong n) const
      {
	if (n < 2) throw InvalidArgumentException(HERE) << "Error: cannot compute recurrence coefficients for n < 2.";
	Coefficients recurrenceCoefficients(3);
	recurrenceCoefficients[0] = 1.0 / sqrt(n);
	recurrenceCoefficients[1] = 0.0;
	recurrenceCoefficients[2] = -sqrt(1.0 - 1.0 / n);
	return recurrenceCoefficients;
      }


      /* String converter */
      String HermiteFactory::__repr__() const
      {
	return OSS() << "class=" << getClassName()
		     << " measure=" << measure_;
      }


    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
