//                                               -*- C++ -*-
/**
 *  @file  FunctionalChaosResult.cxx
 *  @brief The result of a chaos expansion
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "FunctionalChaosResult.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      CLASSNAMEINIT(FunctionalChaosResult);

      static Base::Common::Factory<FunctionalChaosResult> RegisteredFactory("FunctionalChaosResult");


      /* Default constructor */
      FunctionalChaosResult::FunctionalChaosResult()
	: Base::Common::PersistentObject(),
	  model_(),
	  distribution_(),
	  transformation_(),
	  inverseTransformation_(),
	  composedModel_(),
	  orthogonalBasis_(),
	  I_(0),
	  alpha_k_(0),
	  Psi_k_(0),
	  residual_(0.0),
	  metaModel_()
      {
	// Nothing to do
      }

      
      /* Default constructor */
      FunctionalChaosResult::FunctionalChaosResult(const NumericalMathFunction & model,
						   const Distribution & distribution,
						   const NumericalMathFunction & transformation,
						   const NumericalMathFunction & inverseTransformation,
						   const NumericalMathFunction & composedModel,
						   const OrthogonalBasis & orthogonalBasis,
						   const Indices & I,
						   const NumericalPoint & alpha_k,
						   const NumericalMathFunctionCollection & Psi_k,
						   const NumericalScalar residual)
	: Base::Common::PersistentObject(),
	  model_(model),
	  distribution_(distribution),
	  transformation_(transformation),
	  inverseTransformation_(inverseTransformation),
	  composedModel_(composedModel),
	  orthogonalBasis_(orthogonalBasis),
	  I_(I),
	  alpha_k_(alpha_k),
	  Psi_k_(Psi_k),
	  residual_(residual),
	  metaModel_(NumericalMathFunction(Psi_k, alpha_k), transformation)
      {
	// Nothing to do
      }

      
      /* Virtual constructor */
      FunctionalChaosResult * FunctionalChaosResult::clone() const
      {
	return new FunctionalChaosResult(*this);
      }


      /* String converter */
      String FunctionalChaosResult::__repr__() const
      {
	return OSS() << "class=" << getClassName()
		     << " model=" << model_
		     << " distribution=" << distribution_
		     << " transformation=" << transformation_
		     << " inverseTransformation=" << inverseTransformation_
		     << " composedModel=" << composedModel_
		     << " orthogonalBasis=" << orthogonalBasis_
		     << " indices=" << I_
		     << " coefficients=" << alpha_k_
		     << " reduced basis=" << Psi_k_
		     << " residual=" << residual_
		     << " metaModel=" << metaModel_;
      }

	/* Model accessor */
	FunctionalChaosResult::NumericalMathFunction FunctionalChaosResult::getModel() const
	{
	  return model_;
	}

	/* Distribution accessor */
	FunctionalChaosResult::Distribution FunctionalChaosResult::getDistribution() const
	{
	  return distribution_;
	}

	/* IsoProbabilisticTransformation accessor */
	FunctionalChaosResult::NumericalMathFunction FunctionalChaosResult::getTransformation() const
	{
	  return transformation_;
	}

	/* InverseIsoProbabilisticTransformation accessor */
	FunctionalChaosResult::NumericalMathFunction FunctionalChaosResult::getInverseTransformation() const
	{
	  return inverseTransformation_;
	}

	/* Composed model accessor */
	FunctionalChaosResult::NumericalMathFunction FunctionalChaosResult::getComposedModel() const
	{
	  return composedModel_;
	}

	/* Orthogonal basis accessor */
	OrthogonalBasis FunctionalChaosResult::getOrthogonalBasis() const
	{
	  return orthogonalBasis_;
	}

	/* Indices accessor */
	FunctionalChaosResult::Indices FunctionalChaosResult::getIndices() const
	{
	  return I_;
	}

	/* Coefficients accessor */
	FunctionalChaosResult::NumericalPoint FunctionalChaosResult::getCoefficients() const
	{
	  return alpha_k_;
	}

	/* Reduced basis accessor */
	FunctionalChaosResult::NumericalMathFunctionCollection FunctionalChaosResult::getReducedBasis() const
	{
	  return Psi_k_;
	}

	/* Residual accessor */
	NumericalScalar FunctionalChaosResult::getResidual() const
	{
	  return residual_;
	}

	/* MetaModel accessor */
	FunctionalChaosResult::NumericalMathFunction FunctionalChaosResult::getMetaModel() const
	{
	  return metaModel_;
	}


      /* Method save() stores the object through the StorageManager */
      void FunctionalChaosResult::save(const StorageManager::Advocate & adv) const
      {
	PersistentObject::save(adv);
      }


      /* Method load() reloads the object from the StorageManager */
      void FunctionalChaosResult::load(const StorageManager::Advocate & adv)
      {
	PersistentObject::load(adv);
      }



    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
