//                                               -*- C++ -*-
/**
 *  @file  NatafEllipticalDistributionEvaluation.cxx
 *  @brief Class for the Nataf transformation evaluation for elliptical
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: NatafEllipticalDistributionEvaluation.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include "OTprivate.hxx"
#include "NatafEllipticalDistributionEvaluation.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {


      /*
       * @class NatafEllipticalDistributionEvaluation
       *
       * This class offers an interface for the Nataf function for elliptical distributions
       */

      CLASSNAMEINIT(NatafEllipticalDistributionEvaluation);

      /* Parameter constructor */
      NatafEllipticalDistributionEvaluation::NatafEllipticalDistributionEvaluation(const NumericalPoint & mean,
										   const SquareMatrix & inverseCholesky):
	LinearNumericalMathEvaluationImplementation(
						    mean,
						    NumericalPoint(mean.getDimension(), 0.0),
						    inverseCholesky.transpose()
						    )
      {
	// Nothing to do
      }

      /* String converter */
      String NatafEllipticalDistributionEvaluation::__repr__() const
      {
	OSS oss;
	oss << "class=" << NatafEllipticalDistributionEvaluation::GetClassName()
            << " mean=" << getCenter()
            << " inverseCholesky=" << getLinear().transpose();
                   
	return oss;
      }

      /* Gradient according to the marginal parameters.
       * For the Nataf transformation based on elliptical distributions,
       * the marginal parameters \theta are the mean vector \mu and the
       * marginal standard deviations \sigma, which are the reciprocal
       * of the diagonal terms of the inverse Cholesky factor \Gamma of
       * the covariance matrix: \theta = (\mu, \sigma)
       * The expression of the transformation is:
       *
       * T_i(X) = \sum_{j=1}^i \Gamma_{ij}(X_j - mu_j)
       *
       * thus we have:
       *
       * dT_i/dmu_j = -Gamma_{ij} for j>=i,
       *            = 0 otherwise
       * dT_i/dsigma_j = -Gamma_{ii}^2(X_i - mu_i) for j=i,
       *               = 0 otherwise
       *
       * The (i,j)th element of the gradient is dT_j/dtheta_i.
       *
       */

      NatafEllipticalDistributionEvaluation::Matrix NatafEllipticalDistributionEvaluation::parametersGradient(const NumericalPoint & in) const
      {
	NumericalPoint center(getCenter()); // \mu
	Matrix linear(getLinear()); // \Gamma^t, transposed inverse Cholesky 
	UnsignedLong inputDimension(getInputNumericalPointDimension());
	Matrix result(2 * inputDimension, inputDimension);
	// dT_j/dmu_i
	for (UnsignedLong i = 0; i < inputDimension; ++i)
	  {
	    for (UnsignedLong j = i; j < inputDimension; ++j)
	      {
		result(i, j) = -linear(i, j);
	      }
	  }
	// dTj/dsigma_j
	for (UnsignedLong i = 0; i < inputDimension; ++i)
	  {
	    result(inputDimension + i, i) = -linear(i, i) * linear(i, i) * (in[i] - center[i]);
	  }
	return result;
      }
	  
    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
