//                                               -*- C++ -*-
/**
 *  @file  LinearModel.hxx
 *  @brief LinearModel implements the linear model
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: LinearModel.hxx 1262 2009-05-28 12:47:53Z dutka $
 */
#ifndef OPENTURNS_LINEARMODEL_HXX
#define OPENTURNS_LINEARMODEL_HXX

#include "PersistentObject.hxx"
#include "Description.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"
#include "Exception.hxx"
#include "ConfidenceInterval.hxx"
#include "PersistentCollection.hxx"
#include "TestResult.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Stat
    {

      /**
       * @class LinearModel
       *
       * LinearModel implements the notion of linear model
       */

      class LinearModel :
	public Common::PersistentObject
      {
	CLASSNAME;

      public:
        
	typedef Common::InvalidArgumentException                     InvalidArgumentException;
	typedef Type::Description                                    Description;
        typedef Type::NumericalPoint                                 NumericalPoint;
        typedef Type::PersistentCollection<Stat::ConfidenceInterval> ConfidenceIntervalPersistentCollection;
        typedef Type::PersistentCollection<NumericalScalar>          NumericalScalarPersistentCollection;
	typedef Common::StorageManager                               StorageManager;


	/** Default constructor is private */
	LinearModel();

	/** Standard constructor */
	LinearModel(const NumericalPoint & vectorR,
		    const ConfidenceIntervalPersistentCollection & intervalsColl,
		    const NumericalScalarPersistentCollection & pValuesOfR) throw(InvalidArgumentException);
	
	/** Constructor from NumericalPoint */
	LinearModel(const NumericalPoint & vectorR);

	/** Virtual constructor */
	virtual LinearModel * clone() const;


	/** String converter */
	String __repr__() const;
	
	/** get vector, get intervals*/
	NumericalPoint getRegression() const;
	ConfidenceIntervalPersistentCollection getConfidenceIntervals() const;
	NumericalScalarPersistentCollection getPValues() const;
	
	/** getPredicted : build an sample of predicted values */
	NumericalSample getPredicted(const NumericalSample & predictor) const throw(InvalidArgumentException);
	
	/** getResidual */
	NumericalSample getResidual(const NumericalSample & predictor,
                                    const NumericalSample & measured) const throw(InvalidArgumentException);
	
 	/** Method save() stores the object through the StorageManager */
	void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(const StorageManager::Advocate & adv);


      protected:

        /** The estimated vector */
	NumericalPoint regression_;
	
	/** The confidence intervals */
	ConfidenceIntervalPersistentCollection confidenceIntervals_;
	
	/** The p-Values  */
	NumericalScalarPersistentCollection pValues_;
      

      private:

	friend class Common::Factory<LinearModel>;

      }; /* class LinearModel */


      /** Comparison operator */
      Bool operator ==(const LinearModel & lhs, const LinearModel & rhs);  

    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_LINEARMODEL_HXX */
