//                                               -*- C++ -*-
/**
 * @file  CorrelationAnalysis.cxx
 * @brief CorrelationAnalysis implements the sensitivity analysis methods based on correlation coefficients
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-04-08 18:07:10 +0200 (mer. 08 avril 2009) $
 *  Id:      $Id: CorrelationAnalysis.cxx 1211 2009-04-08 16:07:10Z dutka $
 */
#include <cmath>

#include "CorrelationAnalysis.hxx"
#include "Exception.hxx"
#include "LinearModelFactory.hxx"
#include "LinearModel.hxx"

namespace OpenTURNS
{
  namespace Base
  {
    namespace Stat
    {

      typedef Common::InvalidArgumentException             InvalidArgumentException;
      typedef Common::InvalidDimensionException            InvalidDimensionException;
      typedef Stat::CorrelationAnalysis::SobolIndiceResult SobolIndiceResult;
      typedef Type::NumericalPoint                         NumericalPoint;
      typedef Type::SymmetricMatrix                        SymmetricMatrix;
      typedef Type::SymmetricTensor                        SymmetricTensor;

      /* Default constructor */
      CorrelationAnalysis::CorrelationAnalysis() {}

      /* Compute the Pearson correlation coefficient between the component number index of the input sample and the 1D outputSample */
      NumericalScalar CorrelationAnalysis::PearsonCorrelation(const NumericalSample & inputSample,
							      const NumericalSample & outputSample,
							      const UnsignedLong index)
      {
	if (index >= inputSample.getDimension()) throw InvalidArgumentException(HERE) << "Error: given index out of bound";
	if (outputSample.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: output sample must be 1D";
	if (inputSample.getSize() != outputSample.getSize()) throw InvalidArgumentException(HERE) << "Error: input and output samples must have the same size";
	UnsignedLong size(inputSample.getSize());
	NumericalSample pairedSample(size, 2);
	for (UnsignedLong i = 0; i < size; i++)
	  {
	    pairedSample[i][0] = inputSample[i][index];
	    pairedSample[i][1] = outputSample[i][0];
	  }
	return pairedSample.computePearsonCorrelation()(0, 1);
      }

      /* Compute the Spearman correlation coefficient between the component number index of the input sample and the 1D outputSample */
      NumericalScalar CorrelationAnalysis::SpearmanCorrelation(const NumericalSample & inputSample,
							       const NumericalSample & outputSample,
							       const UnsignedLong index)
      {
	if (index >= inputSample.getDimension()) throw InvalidArgumentException(HERE) << "Error: given index out of bound";
	if (outputSample.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: output sample must be 1D";
	if (inputSample.getSize() != outputSample.getSize()) throw InvalidArgumentException(HERE) << "Error: input and output samples must have the same size";
	return PearsonCorrelation(inputSample.rank(), outputSample.rank());
      }

      /* Compute the Standard Regression Coefficients (SRC) between the input sample and the output sample */
      CorrelationAnalysis::NumericalPoint CorrelationAnalysis::SRC(const Base::Stat::NumericalSample & inputSample,
								   const Base::Stat::NumericalSample & outputSample)
      {
	if (outputSample.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: output sample must be 1D";
	if (inputSample.getSize() != outputSample.getSize()) throw InvalidArgumentException(HERE) << "Error: input and output samples must have the same size";
	UnsignedLong dimension(inputSample.getDimension());
	LinearModel linear(LinearModelFactory().buildLM(inputSample, outputSample));
	NumericalPoint regression(linear.getRegression());
	NumericalScalar varOutput(outputSample.computeVariancePerComponent()[0]);
	NumericalPoint src(inputSample.computeVariancePerComponent());
	for (UnsignedLong i = 0; i < dimension; i++)
	  {
	    src[i] *= regression[i + 1] * regression[i + 1] / varOutput;
	  }
	return src;
      }

      /* Compute the Partial Correlation Coefficients (PCC) between the input sample and the output sample */
      CorrelationAnalysis::NumericalPoint CorrelationAnalysis::PCC(const NumericalSample & inputSample,
								   const NumericalSample & outputSample)
      {
	if (inputSample.getDimension() < 2) throw InvalidDimensionException(HERE) << "Error: input sample must have dimension > 1";
	if (outputSample.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: output sample must be 1D";
	if (inputSample.getSize() != outputSample.getSize()) throw InvalidArgumentException(HERE) << "Error: input and output samples must have the same size";
	UnsignedLong dimension(inputSample.getDimension());
	UnsignedLong size(inputSample.getSize());
	NumericalPoint pcc(dimension);
	// For each component i, perform an analysis on the truncated input sample where Xi has been removed
	NumericalSample truncatedInput(size, dimension - 1);
	NumericalSample remainingInput(size, 1);
	for (UnsignedLong index = 0; index < dimension; index++)
	  {
	    // Build the truncated sample
	    for (UnsignedLong i = 0; i < size; i++)
	      {
		for (UnsignedLong j = 0; j < index; j++)
		  {
		    truncatedInput[i][j] = inputSample[i][j];
		  }
		for (UnsignedLong j = index + 1; j < dimension; j++)
		  {
		    truncatedInput[i][j - 1] = inputSample[i][j];
		  }
		remainingInput[i][0] = inputSample[i][index];
	      }
	    // Build the linear models
	    LinearModel outputVersusTruncatedInput(LinearModelFactory().buildLM(truncatedInput, outputSample));
	    LinearModel remainingVersusTruncatedInput(LinearModelFactory().buildLM(truncatedInput, remainingInput));
	    // Compute the correlation between the residuals
	    NumericalSample residualOutput(outputVersusTruncatedInput.getResidual(truncatedInput, outputSample));
	    NumericalSample residualRemaining(remainingVersusTruncatedInput.getResidual(truncatedInput, remainingInput));
	    pcc[index] = PearsonCorrelation(residualOutput, residualRemaining);
	  }
	return pcc;
      }

      /* Compute the Standard Rank Regression Coefficients (SRRC) between the input sample and the output sample */
      CorrelationAnalysis::NumericalPoint CorrelationAnalysis::SRRC(const NumericalSample & inputSample,
								    const NumericalSample & outputSample)
      {
	if (outputSample.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: output sample must be 1D";
	if (inputSample.getSize() != outputSample.getSize()) throw InvalidArgumentException(HERE) << "Error: input and output samples must have the same size";
	return SRC(inputSample.rank(), outputSample.rank());
      }

      /* Compute the Partial Rank Correlation Coefficients (PRCC) between the input sample and the output sample */
      CorrelationAnalysis::NumericalPoint CorrelationAnalysis::PRCC(const NumericalSample & inputSample,
								    const NumericalSample & outputSample)
      {
	// Perform the basic checks of the inputs, to avoid costly ranking if finally PCC will fail
	if (inputSample.getDimension() < 2) throw InvalidDimensionException(HERE) << "Error: input sample must have dimension > 1";
	if (outputSample.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: output sample must be 1D";
	if (inputSample.getSize() != outputSample.getSize()) throw InvalidArgumentException(HERE) << "Error: input and output samples must have the same size";
	return PCC(inputSample.rank(), outputSample.rank());
      }


      /** compute the Sobol' indices given two input samples and a 1d function */
      SobolIndiceResult CorrelationAnalysis::SobolIndice(const SobolIndiceParameters & sobolIndiceParameters, NumericalSample & firstInputSample, NumericalSample & secondInputSample, const NumericalMathFunction & function)
      {
	UnsignedLong size(firstInputSample.getSize());
	UnsignedLong dimension(function.getInputNumericalPointDimension());

	if(firstInputSample.getSize() != secondInputSample.getSize())
	  throw InvalidDimensionException(HERE) << "Error: input samples must have the same size";

	if(firstInputSample.getDimension() != dimension || secondInputSample.getDimension() != dimension)
	  throw InvalidDimensionException(HERE) << "Error: input samples dimension must fit the function input dimension";

	if(function.getOutputNumericalPointDimension() != 1)
	  throw InvalidDimensionException(HERE) << "Error: function output must be 1D";

	if(sobolIndiceParameters.getMaximumOrder() > dimension)
	  throw InvalidDimensionException(HERE) << "Error: indice order cannot exceed input dimension";

	if((sobolIndiceParameters.getMaximumOrder() < 1) && (!sobolIndiceParameters.getTotalIndiceComputation()))
	  throw InvalidDimensionException(HERE) << "Error: none indice to compute";

	// stores the swaped variables values
	NumericalPoint swapPoint(dimension);

	// used to generate the combinations of indexes
	UnsignedLong index[dimension];

	// mean and variance estimates
	NumericalScalar meanEstimate(0), varianceEstimate(0);

	// declare indices
	NumericalPoint firstOrderIndice;
	SymmetricMatrix secondOrderIndice;
	SymmetricTensor thirdOrderIndice;
	NumericalPoint totalOrderIndice;

	// allocate first order indices
	if(sobolIndiceParameters.getMaximumOrder() >= 1)
	  firstOrderIndice = NumericalPoint(dimension, 0.0);

	// allocate second order indices
	if(sobolIndiceParameters.getMaximumOrder() >= 2)
	  secondOrderIndice = SymmetricMatrix(dimension);

	// allocate second order indices
	if(sobolIndiceParameters.getMaximumOrder() >= 3)
	  thirdOrderIndice = SymmetricTensor(dimension, dimension);

	// allocate total order indices
	if(sobolIndiceParameters.getTotalIndiceComputation())
	  totalOrderIndice = NumericalPoint (dimension, 0.0);

	for(UnsignedLong i = 0; i < size; ++i)
	  {
	    // evaluate function on first input sample to estimate meanEstimate and varianceEstimate
	    NumericalScalar firstInputSampleEvaluation(function(firstInputSample[i])[0]);

	    meanEstimate += firstInputSampleEvaluation;
	    varianceEstimate += firstInputSampleEvaluation * firstInputSampleEvaluation;

	    // first order quadratic meanEstimate
	    if(sobolIndiceParameters.getMaximumOrder() >= 1)
	      {
		for(index[0] = 0; index[0] < dimension; ++index[0])
		  {
		    // save value
		    swapPoint[index[0]] = secondInputSample[i][index[0]];

		    // swap variable from first input sample
		    secondInputSample[i][index[0]] = firstInputSample[i][index[0]];

		    // compute quadratic meanEstimate for first order indices
		    firstOrderIndice[index[0]] += firstInputSampleEvaluation * function(secondInputSample[i])[0];

		    // restore value
		    secondInputSample[i][index[0]] = swapPoint[index[0]];
		  }
	      } // end first order

		// second order quadratic meanEstimate
	    if(sobolIndiceParameters.getMaximumOrder() >= 2)
	      {
		for(index[0] = 0; index[0] < dimension; ++index[0])
		  {
		    for(index[1] = index[0] + 1; index[1] < dimension; ++index[1])
		      {
			// save value
			swapPoint[index[0]] = secondInputSample[i][index[0]];
			swapPoint[index[1]] = secondInputSample[i][index[1]];

			// swap variable from first input sample
			secondInputSample[i][index[0]] = firstInputSample[i][index[0]];
			secondInputSample[i][index[1]] = firstInputSample[i][index[1]];

			// compute quadratic meanEstimate for first order indices
			secondOrderIndice(index[0], index[1]) += firstInputSampleEvaluation * function(secondInputSample[i])[0];

			// restore value
			secondInputSample[i][index[0]] = swapPoint[index[0]];
			secondInputSample[i][index[1]] = swapPoint[index[1]];
		      }
		  }
	      } // end second order

		// third order quadratic meanEstimate
	    if(sobolIndiceParameters.getMaximumOrder() >= 3)
	      {
		for(index[0] = 0; index[0] < dimension; ++index[0])
		  {
		    for(index[1] = index[0] + 1; index[1] < dimension; ++index[1])
		      {
			for(index[2] = index[1] + 1; index[2] < dimension; ++index[2])
			  {
			    // save value
			    swapPoint[index[0]] = secondInputSample[i][index[0]];
			    swapPoint[index[1]] = secondInputSample[i][index[1]];
			    swapPoint[index[2]] = secondInputSample[i][index[2]];
	
			    // swap variable from first input sample
			    secondInputSample[i][index[0]] = firstInputSample[i][index[0]];
			    secondInputSample[i][index[1]] = firstInputSample[i][index[1]];
			    secondInputSample[i][index[2]] = firstInputSample[i][index[2]];
	
			    // compute quadratic meanEstimate for first order indices
			    thirdOrderIndice(index[0], index[1], index[2]) += firstInputSampleEvaluation * function(secondInputSample[i])[0];
	
			    // restore value
			    secondInputSample[i][index[0]] = swapPoint[index[0]];
			    secondInputSample[i][index[1]] = swapPoint[index[1]];
			    secondInputSample[i][index[2]] = swapPoint[index[2]];
			  }
		      }
		  }
	      } // end third order

		// total order quadratic meanEstimate
	    if(sobolIndiceParameters.getTotalIndiceComputation())
	      {
		for(index[0] = 0; index[0] < dimension; ++index[0])
		  {
		    // save value
		    swapPoint[index[0]] = firstInputSample[i][index[0]];

		    // swap variable from second input sample
		    firstInputSample[i][index[0]] = secondInputSample[i][index[0]];

		    // compute quadratic meanEstimate for first order indices
		    totalOrderIndice[index[0]] += firstInputSampleEvaluation * function(firstInputSample[i])[0];

		    // restore value
		    firstInputSample[i][index[0]] = swapPoint[index[0]];
		  }
	      } // end total order

	  } // end for i

	// uppdate mean and variance estimates
	meanEstimate /= size;
	varianceEstimate /= size;
	varianceEstimate -= meanEstimate * meanEstimate;
	
	// calculate indices from quadratic conditionnal meanEstimate
	if(sobolIndiceParameters.getMaximumOrder() >= 1)
	  {
	    for( index[0] = 0; index[0] < dimension; ++index[0])
	      {
		firstOrderIndice[index[0]] = (firstOrderIndice[index[0]] / size - meanEstimate * meanEstimate) / varianceEstimate;
	      }
	  } // end first order

	if(sobolIndiceParameters.getMaximumOrder() >= 2)
	  {
	    for( index[0] = 0; index[0] < dimension; ++index[0])
	      {
		for(index[1] = index[0] + 1; index[1] < dimension; ++index[1])
		  {
		    secondOrderIndice(index[0], index[1]) = (secondOrderIndice(index[0], index[1]) / size - meanEstimate * meanEstimate) / varianceEstimate;
		    secondOrderIndice(index[0], index[1]) -= firstOrderIndice[index[0]] + firstOrderIndice[index[1]];
		  }
	      }
	  } // end second order

	if(sobolIndiceParameters.getMaximumOrder() >= 3)
	  {
	    for(index[0] = 0; index[0] < dimension; ++index[0])
	      {
		for(index[1] = index[0] + 1; index[1] < dimension; ++index[1])
		  {
		    for(index[2] = index[1] + 1; index[2] < dimension; ++index[2])
		      {
			thirdOrderIndice(index[0], index[1], index[2]) = (thirdOrderIndice(index[0], index[1], index[2]) / size - meanEstimate * meanEstimate) / varianceEstimate;
			thirdOrderIndice(index[0], index[1], index[2]) -= secondOrderIndice(index[0], index[1]) + secondOrderIndice(index[0], index[2]) + secondOrderIndice(index[1], index[2]);
			thirdOrderIndice(index[0], index[1], index[2]) -= firstOrderIndice[index[0]] + firstOrderIndice[index[1]] + firstOrderIndice[index[2]];
		      }
		  }
	      }
	  } // end third order

	if(sobolIndiceParameters.getTotalIndiceComputation())
	  {
	    for( index[0] = 0; index[0] < dimension; ++index[0])
	      {
		totalOrderIndice[index[0]] = 1.0 - (totalOrderIndice[index[0]] / size - meanEstimate * meanEstimate) / varianceEstimate;
	      }
	  } // end total order

	// create a SobolIndiceResult object to return all the indices
	SobolIndiceResult result;
	result.setFirstOrderIndice(firstOrderIndice);
	result.setSecondOrderIndice(secondOrderIndice);
	result.setThirdOrderIndice(thirdOrderIndice);
	result.setTotalOrderIndice(totalOrderIndice);
	return result;
      }


    } // namespace Stat
  } // namespace Base
} // namespace OpenTURNS
