//                                               -*- C++ -*-
/**
 *  @file  LeastSquaresAlgorithm.cxx
 *  @brief This Least Square Algorithm as a functor class
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "LeastSquaresAlgorithm.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"
#include "Matrix.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Optim
    {

      CLASSNAMEINIT(LeastSquaresAlgorithm);

      static Base::Common::Factory<LeastSquaresAlgorithm> RegisteredFactory("LeastSquaresAlgorithm");

      typedef Type::Matrix                     Matrix;
      typedef Common::InvalidArgumentException InvalidArgumentException;

      /* Default constructor */
      LeastSquaresAlgorithm::LeastSquaresAlgorithm()
	: Base::Common::PersistentObject()
      {
	// Nothing to do
      }

      /* Parameters constructor */
      LeastSquaresAlgorithm::LeastSquaresAlgorithm(const NumericalSample & inputSample,
						   const NumericalSample & outputSample,
						   const NumericalPoint & weight,
						   const NumericalMathFunctionCollection & basis):
	inputSample_(inputSample),
	outputSample_(outputSample),
	weight_(weight),
	basis_(basis),
	coefficients_(0),
	residual_(0.0),
	isAlreadyComputedCoefficients_(false)
      {
	const UnsignedLong size(inputSample.getSize());
	if (size == 0) throw InvalidArgumentException(HERE) << "Error: cannot perform least squares approximation based on an empty sample.";
	if (size != outputSample.getSize()) throw InvalidArgumentException(HERE) << "Error: cannot perform least squares approximation based on samples of different size.";
	if (outputSample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: cannot perform least squares approximation based on output sample of dimension different from 1.";
	if (size != weight.getDimension()) throw InvalidArgumentException(HERE) << "Error: cannot perform least squares approximation with weight of improper dimension.";
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    if (weight[i] < 0.0) throw InvalidArgumentException(HERE) << "Error: can only use positive weight.";
	  }
	for (UnsignedLong i = 0; i < basis.getSize(); ++i)
	  {
	    if (basis[i].getOutputNumericalPointDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can only use basis functions with 1D values.";
	  }
      }

      /* Parameters constructor */
      LeastSquaresAlgorithm::LeastSquaresAlgorithm(const NumericalSample & inputSample,
						   const NumericalSample & outputSample,
						   const NumericalMathFunctionCollection & basis):
	inputSample_(inputSample),
	outputSample_(outputSample),
	weight_(inputSample.getSize(), 1.0),
	basis_(basis),
	coefficients_(0),
	residual_(0.0),
	isAlreadyComputedCoefficients_(false)
      {
	const UnsignedLong size(inputSample.getSize());
	if (size == 0) throw InvalidArgumentException(HERE) << "Error: cannot perform least squares approximation based on an empty sample.";
	if (size != outputSample.getSize()) throw InvalidArgumentException(HERE) << "Error: cannot perform least squares approximation based on samples of different size.";
	if (outputSample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: cannot perform least squares approximation based on output sample of dimension different from 1.";
	for (UnsignedLong i = 0; i < basis.getSize(); ++i)
	  {
	    if (basis[i].getOutputNumericalPointDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can only use basis functions with 1D values.";
	  }
      }

      /* Coefficients accessor */
      LeastSquaresAlgorithm::NumericalPoint LeastSquaresAlgorithm::getCoefficients()
      {
	if (!isAlreadyComputedCoefficients_) run();
	return coefficients_;
      }
      
      /* Residual accessor */
      NumericalScalar LeastSquaresAlgorithm::getResidual()
      {
	if (!isAlreadyComputedCoefficients_) run();
	return residual_;
      }
      
      /* Perform the optimization */
      void LeastSquaresAlgorithm::run()
      {
	const UnsignedLong basisDimension(basis_.getSize());
	const UnsignedLong sampleSize(inputSample_.getSize());
	Matrix basisMatrix(sampleSize, basisDimension);
	NumericalPoint secondMember(sampleSize);
	NumericalPoint weightSquareRoot(sampleSize);
	// Build the second member
	for (UnsignedLong rowIndex = 0; rowIndex < sampleSize; ++rowIndex)
	  {
	    weightSquareRoot[rowIndex] = sqrt(weight_[rowIndex]);
	    secondMember[rowIndex] = weightSquareRoot[rowIndex] * outputSample_[rowIndex][0];
	  }

	// Build the matrix column-wise
	for (UnsignedLong functionIndex = 0; functionIndex < basisDimension; ++functionIndex)
	  {
	    NumericalSample functionSample(basis_[functionIndex](inputSample_));
	    // Fill-in the matrix column
	    for (UnsignedLong rowIndex = 0; rowIndex < sampleSize; ++rowIndex)
	      {
		basisMatrix(rowIndex, functionIndex) = functionSample[rowIndex][0] * weightSquareRoot[rowIndex];
	      }
	  }
	coefficients_ = basisMatrix.solveLinearSystem(secondMember);
	residual_ = (basisMatrix * coefficients_ - secondMember).norm() / secondMember.getDimension();
	isAlreadyComputedCoefficients_ = true;
      }


      /* Virtual constructor */
      LeastSquaresAlgorithm * LeastSquaresAlgorithm::clone() const
      {
	return new LeastSquaresAlgorithm(*this);
      }


      /* String converter */
      String LeastSquaresAlgorithm::__repr__() const
      {
	return OSS() << "class=" << getClassName()
		     << " inputSample=" << inputSample_
		     << " outputSample=" << outputSample_
		     << " weight=" << weight_
		     << " basis=" << basis_;
      }


      /* Method save() stores the object through the StorageManager */
      void LeastSquaresAlgorithm::save(const StorageManager::Advocate & adv) const
      {
	PersistentObject::save(adv);
      }


      /* Method load() reloads the object from the StorageManager */
      void LeastSquaresAlgorithm::load(const StorageManager::Advocate & adv)
      {
	PersistentObject::load(adv);
      }



    } /* namespace Optim */
  } /* namespace Base */
} /* namespace OpenTURNS */
