//                                               -*- C++ -*-
/**
 * @file  QuadraticNumericalMathEvaluationImplementation.hxx
 * @brief Class for a quadratic numerical math function implementation
 *        of the form y = constant + <linear, x-c> + <<quadratic, x-c>, x-c>
 *        where constant is a dim(y) numerical point, c a dim(x) numerical
 *        point, linear is a dim(x) by dim(y) matrix, quadratic is a
 *        dim(x) by dim(x) by dim(y) symmetric tensor and <linear, x>
 *        means Transpose(linear).x, <quadratic, x> means
 *        Transpose_kj(quadratic).x
 *
 * (C) Copyright 2005-2006 EADS
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * \author $LastChangedBy: dutka $
 * \date   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 */

#ifndef OPENTURNS_QUADRATICNUMERICALMATHEVALUATIONIMPLEMENTATION_HXX
#define OPENTURNS_QUADRATICNUMERICALMATHEVALUATIONIMPLEMENTATION_HXX

#include "NumericalMathEvaluationImplementation.hxx"
#include "Exception.hxx"
#include "NumericalPoint.hxx"
#include "Matrix.hxx"
#include "SymmetricTensor.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {


      /**
       * @class QuadraticNumericalMathEvaluationImplementation
       *
       * This class offers an interface for an real numerical mathematical evaluation
       * implementations into the platform. It realizes the computation of the image vector
       * of a vector (aka NumericalPoint) through the evaluation. The
       * evaluation may be loaded in an external way through a wrapper.
       */
      class QuadraticNumericalMathEvaluationImplementation
	: public NumericalMathEvaluationImplementation
      {
	CLASSNAME;
      public:

	typedef NumericalMathEvaluationImplementation::NumericalPoint           NumericalPoint;
	typedef NumericalMathEvaluationImplementation::Matrix                   Matrix;
	typedef NumericalMathEvaluationImplementation::InvalidArgumentException InvalidArgumentException;
	typedef NumericalMathEvaluationImplementation::InternalException        InternalException;
	typedef Type::SymmetricTensor                                           SymmetricTensor;
	typedef Common::InvalidDimensionException                               InvalidDimensionException;
	typedef NumericalMathEvaluationImplementation::StorageManager           StorageManager;

	/** Default constructor */
	QuadraticNumericalMathEvaluationImplementation();

	/** Parameter constructor */
	QuadraticNumericalMathEvaluationImplementation(const NumericalPoint & center,
						       const NumericalPoint & constant,
						       const Matrix & linear,
						       const SymmetricTensor & quadratic) throw(InvalidDimensionException);


	/** Virtual constructor */
	virtual QuadraticNumericalMathEvaluationImplementation * clone() const;

	/** Comparison operator */
	Bool operator ==(const QuadraticNumericalMathEvaluationImplementation & other) const;

	/** String converter */
	virtual String __repr__() const;



	/* Here is the interface that all derived class must implement */
	
	/** Operator () */
	virtual NumericalPoint operator() (const NumericalPoint & in) const 
          throw(InvalidArgumentException, InternalException);

	/** Accessor for input point dimension */
	virtual UnsignedLong getInputNumericalPointDimension() const
          throw(InternalException);

	/** Accessor for output point dimension */
	virtual UnsignedLong getOutputNumericalPointDimension() const
	  throw(InternalException);

	/** Accessor for the center */
	NumericalPoint getCenter() const;

	/** Accessor for the constant term */
	NumericalPoint getConstant() const;

	/** Accessor for the linear term */
	Matrix getLinear() const;

	/** Accessor for the quadratic term */
	SymmetricTensor getQuadratic() const;

 	/** Method save() stores the object through the StorageManager */
	void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(const StorageManager::Advocate & adv);

      protected:

    
      private:
	NumericalPoint center_;
	NumericalPoint constant_;
	Matrix linear_;
	SymmetricTensor quadratic_;
      }; /* class QuadraticNumericalMathEvaluationImplementation */


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_QUADRATICNUMERICALMATHEVALUATIONIMPLEMENTATION_HXX */
