//                                               -*- C++ -*-
/**
 *  @file  XMLToolbox.cxx
 *  @brief This file provides basic XML functionalities
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-17 14:26:02 +0200 (Fri, 17 Oct 2008) $
 *  Id:      $Id: WrapperFile.cxx 974 2008-10-17 12:26:02Z dutka $
 */
#include "OTprivate.hxx"
#include "OSS.hxx"
#include "Log.hxx"
#include "Exception.hxx"
#include "XMLToolbox.hxx"
#include <cstdio>
#include <cstdlib>
#include <cstdarg>
#include <cassert>

#if defined HAVE_LIBXML2
#include <libxml/tree.h>
#include <libxml/xmlsave.h>
#endif

namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {

      using Common::WrapperFileParsingException;
      using Common::Log;


#if defined HAVE_LIBXML2

      XMLDoc::XMLDoc() : doc_(xmlNewDoc( REINTERPRET_CAST( const xmlChar *, "1.0" ) ))
      {
	// Nothing to do
      }

      XMLDoc::XMLDoc(const XMLDoc & other) : doc_(xmlCopyDoc( other.doc_, 1 ))
      {
	// Nothing to do
      }

      XMLDoc::XMLDoc(const FileName & pathToFile) : doc_(0)
      {
	doc_ = xmlParseFile( pathToFile.c_str() );
	if (doc_ == NULL) throw WrapperFileParsingException(HERE) << "Error in parsing wrapper file " << pathToFile;
      }

      XMLDoc::XMLDoc(const char * buffer, int size) : doc_(0)
      {
	doc_ = xmlParseMemory( buffer, size );
	if (doc_ == NULL) throw WrapperFileParsingException(HERE) << "Error in parsing buffer";
      }

      XMLDoc::~XMLDoc() throw()
      {
	xmlFreeDoc( doc_ );

      }

      XMLDoc & XMLDoc::operator =(const XMLDoc & other)
      {
	if (this != &other) {
	  xmlFreeDoc( doc_ );
	  doc_ = xmlCopyDoc( other.doc_, 1 );
	}

	return *this;
      }

      XMLDoc::operator xmlDocPtr()
      {
	return doc_;
      }

      XMLDoc::operator const xmlDocPtr() const
      {
	return doc_;
      }

      void XMLDoc::save( const FileName & fileName ) const
      {
	xmlSaveFormatFileEnc(fileName.c_str(), doc_, "UTF-8", 1);
      }

      String XMLDoc::__repr__() const
      {
	xmlSaveCtxt * xctx;
	xmlBuffer * buf;

	// Buffer allocation
	if ( !( buf = xmlBufferCreate() ) )
	  return String();

	// Context definition
	if ( !( xctx = xmlSaveToBuffer( buf, NULL, XML_SAVE_NO_DECL ) ) ) {
	  xmlBufferFree( buf );
	  return String();
	}

	// Document streaming
	if ( xmlSaveDoc( xctx, doc_ ) < 0 ) {
	  xmlSaveClose( xctx );
	  xmlBufferFree( buf );
	  return String();
	}

	// Context closing (?)
	if ( xmlSaveClose( xctx ) <= 0) {
	  xmlBufferFree( buf );
	  return String();
	}

	String data = REINTERPRET_CAST( char * , buf->content );

	xmlBufferFree( buf );

	return data;
      }


      Bool XMLDoc::hasDTD() const
      {
	xmlDtdPtr dtd = xmlGetIntSubset( doc_ );
	return (dtd != NULL);
      }

      Bool XMLDoc::validate() const
      {
	xmlValidCtxtPtr validCtxt = xmlNewValidCtxt();
// 	validCtxt->error    = REINTERPRET_CAST( xmlValidityErrorFunc,   XML::ErrorHandler );
// 	validCtxt->warning  = REINTERPRET_CAST( xmlValidityWarningFunc, XML::ErrorHandler );
	int ok = xmlValidateDocument( validCtxt, doc_ );
	xmlFreeValidCtxt( validCtxt );
	return (ok == 1);
      }

      Bool XMLDoc::validate(const String & name, const FileName & dtd) const
      {
	XML::xmlString aName  = XML::xmlString( name.begin(), name.end() );
	XML::xmlString aDtd   = XML::xmlString( dtd.begin(), dtd.end() );
	xmlDtdPtr theDTD = xmlParseDTD( NULL, aDtd.c_str() );
	xmlValidCtxtPtr validCtxt = xmlNewValidCtxt();
// 	validCtxt->error    = REINTERPRET_CAST( xmlValidityErrorFunc,   XML::ErrorHandler );
// 	validCtxt->warning  = REINTERPRET_CAST( xmlValidityWarningFunc, XML::ErrorHandler );
	int ok = xmlValidateDtd( validCtxt, doc_, theDTD );
	xmlFreeValidCtxt( validCtxt );
	xmlFreeDtd( theDTD );
	return (ok == 1);	
      }




      void XML::ErrorHandler( void * ptr, const char * format, ...)
      {
	int ret = 0;
	va_list args;
	char *msg;

	va_start(args, format);
	msg = MakeMessage( format, args );
	va_end(args);

	Log::Warn( OSS() << msg );
	free(msg);
      }

      
      /* From printf(3) linux manpage */
      char * XML::MakeMessage(const char *fmt, ...)
      {
	/* Guess we need no more than 100 bytes. */
	int n, size = 100;
	char *p, *np;
	va_list ap;
	
	if ((p = (char*) calloc (sizeof(char), size)) == NULL)
	  return NULL;
	
	while (1) {
	  /* Try to print in the allocated space. */
	  va_start(ap, fmt);
	  n = std::vsnprintf (p, size, fmt, ap);
	  va_end(ap);
	  /* If that worked, return the string. */
	  if (n > -1 && n < size)
	    return p;
	  /* Else try again with more space. */
	  if (n > -1)    /* glibc 2.1 */
	    size = n+1; /* precisely what is needed */
	  else           /* glibc 2.0 */
	    size *= 2;  /* twice the old size */
	  if ((np = (char*) realloc (p, size)) == NULL) {
	    free(p);
	    return NULL;
	  } else {
	    p = np;
	  }
	}
      }


      String XML::ToString(const xmlString & st)
      {
	return String(st.begin(), st.end());
      }

      Bool XML::IsText(const Node & elt)
      {
	assert( elt != NULL );
	return (elt->type == XML_TEXT_NODE);
      }



      Bool XML::IsElement(const Node & elt)
      {
	assert( elt != NULL );
	return (elt->type == XML_ELEMENT_NODE);
      }



      Bool XML::IsElement(const Node & elt, const String & name)
      {
	xmlString aName = xmlString( name.begin(), name.end() );
	Bool isElt      = IsElement( elt );
	Bool hasName    = ( xmlStrcmp( elt->name, aName.c_str() ) == 0 );
	return isElt && hasName;
      }



      Bool XML::ElementHasAttribute(const Node & elt, const String & name)
      {
	xmlString aName = xmlString( name.begin(), name.end() );
	return xmlHasProp( elt, aName.c_str() );
      }



      String XML::GetAttributeByName(const Node & node, const String & name)
      {
	String attrVal;
	if (node) {
	  xmlString aName = xmlString( name.begin(), name.end() );
	  if ( xmlHasProp( node, aName.c_str() ) ) {
	    xmlString val = xmlGetProp(node, aName.c_str() );
	    attrVal = String(val.begin(), val.end());
	  }
	}
	return attrVal;
      }


      void XML::SetAttribute(const Node & node, const String & attribute, const String & value)
      {
	if (node) {
	  xmlString aAttr  = xmlString( attribute.begin(), attribute.end() );
	  xmlString aValue = xmlString( value.begin(), value.end() );
	  xmlNewProp( node, aAttr.c_str(), aValue.c_str() );
	}
      }


      XML::Node XML::FindElementByName(const Node & node, const String & name)
      {
	Node cur = NULL;
	if (node) {
	  for(cur = node->children; cur; cur = cur->next) {
	    if (IsElement(cur, name)) {
	      break;
	    }
	  }
	}
	return cur;
      }



      String XML::GetNodeValue(const Node & node)
      {
	String value;
	Node cur = NULL;
	if (node) {
	  for(cur = node->children; cur; cur = cur->next) {
	    if (IsText(cur)) {
	      xmlString val = cur->content;
	      value = String(val.begin(), val.end());
	      break;
	    }
	  }
	}
	return value;
      }


      String XML::GetNodeName(const Node & node)
      {
	String name;
	if (node) {
	  xmlString aName = node->name;
	  name = String( aName.begin(), aName.end() );
	}
	return name;
      }



      UnsignedLong XML::GetNodeLineNumber(const Node & node)
      {
	UnsignedLong lineno = 0;
	if (node) lineno = xmlGetLineNo(node);
	return lineno;
      }


      XML::Node XML::NewNode(const String & name)
      {
	xmlString aName = xmlString( name.begin(), name.end() );
	Node node = xmlNewNode( NULL, aName.c_str() );
	return node;
      }



      XML::Node XML::NewTextNode(const String & value)
      {
	xmlString aValue = xmlString( value.begin(), value.end() );
	Node node = xmlNewText( aValue.c_str() );
	return node;
      }



      XML::Node XML::NewNode(const String & name, const String & value)
      {
	Node node = NewNode( name );
	Node child = NewTextNode( value );
	AddChild( node, child );
	return node;
      }
	

      void XML::AddChild(const Node & parent, const Node & child)
      {
	xmlAddChild( parent, child );
      }

	
      XML::Node XML::GetRootNode( const XMLDoc & doc )
      {
	return xmlDocGetRootElement( doc );
      }


      void XML::SetRootNode( const XMLDoc & doc, const Node & root )
      {
	xmlDocSetRootElement( doc, root );
      }

      XML::Node XML::GetFirstChild( const Node & node )
      {
	return node->children;
      }

      XML::Node XML::GetNextNode( const Node & node )
      {
	return node->next;
      }

      void XML::SetDTD( const XMLDoc & doc, const String & name, const String & path )
      {
	xmlString aName = xmlString( name.begin(), name.end() );
	xmlString aPath = xmlString( path.begin(), path.end() );
	xmlCreateIntSubset( doc, aName.c_str(), NULL, aPath.c_str() );
      }


#endif /* HAVE_LIBXML2 */

    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */
