//                                               -*- C++ -*-
/**
 *  @file  PersistentObjectFactory.hxx
 *  @brief Class PersistentObjectFactory reloads a PersistentObject from a storage manager
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-29 18:54:45 +0100 (mer. 29 oct. 2008) $
 *  Id:      $Id: PersistentObjectFactory.hxx 990 2008-10-29 17:54:45Z dutka $
 */
#ifndef OPENTURNS_PERSISTENTOBJECTFACTORY_HXX
#define OPENTURNS_PERSISTENTOBJECTFACTORY_HXX

#include "PersistentObject.hxx"
#include "StorageManager.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {

      /**
       * @class PersistentObjectFactory
       *
       * Class PersistentObjectFactory reloads a PersistentObject
       *  from a storage manager
       */

      class PersistentObjectFactory
      {
      public:

	typedef Common::StorageManager StorageManager;

	/** Virtual constructor */
	virtual PersistentObjectFactory * clone() const = 0;

	/** Destructor */
	virtual ~PersistentObjectFactory();

	/** Method build() creates a new PersistentObject from the storage manager */
	virtual PersistentObject * build(const StorageManager & mgr) const = 0;

	/** Method assign() fills a PersistentObject with another one (must have same type) */
	virtual void assign(PersistentObject & po, const PersistentObject & other) const = 0;

	/** This method register the factory into the Catalog */
	void registerMe(const String & className) const;

	/** Accessor to PersistentObject's shadowed id */
	void setShadowedId(PersistentObject & obj, Id id) const;
	Id getShadowedId(PersistentObject & obj) const;

      protected:

      private:

      }; /* class PersistentObjectFactory */


      /**
       * This template class ease the creation of factories for any PersistentObject
       * based class
       */
      template <class PERSISTENT>
      class Factory
	: PersistentObjectFactory
      {
      public:
	/** Constructor */
	Factory(const String & className)
	{
	  registerMe(className);
	}

	/** Virtual constructor */
	virtual Factory * clone() const
	{
	  return new Factory(*this);
	}

	/** Method build() creates a new NumericalPointImplementation from the storage manager */
	virtual PersistentObject * build(const StorageManager & mgr) const
	{
	  StorageManager::Advocate adv ( mgr.readObject() );
	  
	  PERSISTENT * p_rebuildObject = new PERSISTENT();
	  try {
	    p_rebuildObject->load(adv);
	  } catch (Exception & ex) {
	    // Nothing to do
	    // The rebuild object remains the default one
	  }
	  return p_rebuildObject;
	}
	
	/** Method assign() fills a PersistentObject with another one (must have same type) */
	virtual void assign(PersistentObject & po, const PersistentObject & other) const
	{
	  PERSISTENT & ref_po          = static_cast<PERSISTENT &>(po);
	  const PERSISTENT & ref_other = static_cast<const PERSISTENT &>(other);
	  ref_po = ref_other;
	}

      }; /* end class Factory */


    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_PERSISTENTOBJECTFACTORY_HXX */
