//                                               -*- C++ -*-
/**
 *  @file  ExceptionDefs.cxx
 *  @brief Exception defines top-most exception strategies
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2009-05-28 14:47:53 +0200 (jeu. 28 mai 2009) $
 *  Id:      $Id: ExceptionDefs.cxx 1262 2009-05-28 12:47:53Z dutka $
 */
#include "ExceptionDefs.hxx"
#include "Exception.hxx"
#include "Log.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {




      /* Default constructor */
      Exception::Exception(const PointInSourceFile & point ///< The location where the exception was raised
			   )
	: point_(point),
	  reason_(),
	  className_("Exception")
      {
	// Nothing to do
      }

      /* Copy constractor */
      Exception::Exception(const Exception & other)
	: point_(other.point_), reason_(), className_(other.className_)
      {
	reason_ << String(other.reason_);
	Log::Debug(OSS() << className_  << " emited at " << point_.str() << ": " << String(reason_) );
      }

      /* Inheritance constructor */
      Exception::Exception(const PointInSourceFile & point, ///< The location where the exception was raised
			   const char * type ///< The classname of the exception
			   )
	: point_(point),
	  reason_(),
	  className_(type)
      {
	// Nothing to do
      }

      /* Inheritance constructor */
      Exception::Exception(const PointInSourceFile & point, ///< The location where the exception was raised
			   int id ///< The id in the global map that gives its name to the exception
			   )
	: point_(point),
	  reason_(),
	  className_(ExceptionNames::get(id))
      {
	// Nothing to do
      }



      /* Destructor */
      Exception::~Exception() throw()
      {
        // Nothing to do
      }


      /* String converter */
      String Exception::__repr__() const throw()
      {
	return OSS() << className_ << " : " << String(reason_);
      }


      /* Point accessor */
      const char * Exception::where() const throw()
      {
	return point_.str().c_str();
      }


      /* Reason accessor */
      const char * Exception::what() const throw()
      {
	return String(reason_).c_str();
      }


      /* Class name accessor */
      const char * Exception::type() const throw()
      {
	return className_.c_str();
      }


      /*
       * @fn std::ostream & operator <<(std::ostream & os, const Exception & obj)
       * @brief Output stream converter
       * @param os A STL output stream exception
       * @param obj The exception read by \em os
       * @return A reference to \em os
       *
       * Operator << converts the Exception object to an output stream
       * so it is easy to show the reason of the exception.
       */
      std::ostream & operator <<(std::ostream & os, const Exception & obj)
      {
        return os << obj.__repr__();
      }

      OStream & operator <<(OStream & OS, const Exception & obj)
      {
        return OS << obj.__repr__();
      }





    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */
