//                                               -*- C++ -*-
/**
 *  @file  t_StrongMaximumTest_std.cxx
 *  @brief The test file of StrongMaximumTest class
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2007-05-02 14:59:25 +0200 (mer 02 mai 2007) $
 *  Id:      $Id: t_StrongMaximumTest_std.cxx 424 2007-05-02 12:59:25Z dutka $
 */
#include <iostream>
#include <cmath>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "StandardEvent.hxx"
#include "RandomVector.hxx"
#include "Less.hxx"
#include "Greater.hxx"
#include "NumericalMathFunction.hxx"
#include "IdentityMatrix.hxx"
#include "Normal.hxx"
#include "StrongMaximumTest.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Func;
using namespace OT::Base::Stat;
using namespace OT::Uncertainty::Model;
using namespace OT::Uncertainty::Distribution;
using namespace OT::Uncertainty::Algorithm;

void printSample(String name, NumericalSample sample)
{
  std::cout << name << "=[" ;
  for (UnsignedLong counter = 0; counter < sample.getSize(); counter++)
    {
      if (counter != 0)  std::cout << ";";
      std::cout << "[" ;
      NumericalPoint point(sample[counter]);	
      for (UnsignedLong coordinate = 0; coordinate < point.getDimension(); coordinate++)
	{
	  if (coordinate != 0)  std::cout << ",";	  
	  std::cout <<  point[coordinate];
	}
      std::cout << "]" ;
    }
  std::cout << "];" << std::endl;
}

int main(int argc, char *argv[])
{
  TESTPREAMBLE;

  try {

    /* We create a numerical math function */
    NumericalMathFunction myFunction("TestStrongMaximumTest");

    UnsignedLong dim(myFunction.getInputNumericalPointDimension());
    /* We create a normal distribution point of dimension 1 */
    NumericalPoint mean(dim, 0.0);
    NumericalPoint sigma(dim, 1.0);
    IdentityMatrix R(dim);
    Normal myDistribution(mean, sigma, R);

    /* We create a 'usual' RandomVector from the Distribution */
    RandomVector vect(myDistribution);

    /* We create a composite random vector */
    RandomVector output(myFunction, vect);

    /* We create a StandardEvent from this RandomVector  : RandomVector > 0.0 */
    double seuil(10);
    StandardEvent myStandardEvent(output, Greater(), seuil);

    /* We create the design point */
    NumericalPoint designPoint(dim, 0.0);
    double C(0.3);
    designPoint[0] = - sqrt(seuil) + C;

    /* We create the "second" design point */
    NumericalPoint pseudoDesignPoint(dim, 0.0);
    pseudoDesignPoint[0] = sqrt(seuil) + C;

    NumericalScalar importanceLevel(0.01);
    NumericalScalar accuracyLevel(2);
    NumericalScalar confidenceLevel(0.999999);

    StrongMaximumTest myTest(myStandardEvent, designPoint, importanceLevel, accuracyLevel, confidenceLevel);
    std::cout << "myTest=" << myTest << std::endl;
    myTest.run();
    std::cout << "Beta = " << designPoint.norm() << std::endl;
    std::cout << "Discretised sphere radius = "  << designPoint.norm() * (1 + myTest.getAccuracyLevel() * myTest.getDeltaEpsilon()) << std::endl;
    std::cout << "PointNumber = " << myTest.getPointNumber() << std::endl;
    std::cout << "DesignPointVicinity Angle (rad)= " << acos(myTest.getDesignPointVicinity()) << std::endl;
    std::cout << "DesignPointVicinity Angle (deg)= " << acos(myTest.getDesignPointVicinity())*180/M_PI << std::endl;
    std::cout << "Near Design Point Verifying Event Points Number = " << (myTest.getNearDesignPointVerifyingEventPoints()).getSize() << std::endl;
    std::cout << "Near Design Point Violating Event Points Number = " << (myTest.getNearDesignPointViolatingEventPoints()).getSize() << std::endl;
    std::cout << "Far Design Point Verifying Event Points Number = " << (myTest.getFarDesignPointVerifyingEventPoints()).getSize() << std::endl;
    std::cout << "Far Design Point Violating Event Points Number = " << (myTest.getFarDesignPointViolatingEventPoints()).getSize() << std::endl;
    std::cout << "//////" << std::endl;
    // parameters of the test
    std::cout << "importanceLevel=" << myTest.getImportanceLevel()<< std::endl;
    std::cout << "accuracyLevel=" << myTest.getAccuracyLevel()<< std::endl;
    std::cout << "confidenceLevel=" << myTest.getConfidenceLevel()<< std::endl;

    // design point coordinates
    std::cout << "standardSpaceDesignPoint=[" << myTest.getStandardSpaceDesignPoint()[0] << "," << myTest.getStandardSpaceDesignPoint()[1] << "]" << std::endl;

    // pseudo design point coordinates
    std::cout << "pseudoStandardSpaceDesignPoint=[" << pseudoDesignPoint[0] << "," << pseudoDesignPoint[1] << "]" << std::endl;

    // cout of the coordinates of the points of the 4 samples
    /* NearDesignPointVerifyingEventPoints */ 
    printSample("NearDesignPointVerifyingEventPointsSample",myTest.getNearDesignPointVerifyingEventPoints());
    /* NearDesignPointViolatingEventPoints */ 
    printSample("NearDesignPointViolatingEventPoints",myTest.getNearDesignPointViolatingEventPoints());
    /* FarDesignPointVerifyingEventPoints */ 
    printSample("FarDesignPointVerifyingEventPoints",myTest.getFarDesignPointVerifyingEventPoints());
    /* FarDesignPointViolatingEventPoints */ 
    printSample("FarDesignPointViolatingEventPoints",myTest.getFarDesignPointViolatingEventPoints());

    //    std::cout << "myTest (after run())=" << myTest << std::endl;
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
