//                                               -*- C++ -*-
/**
 *  @file  t_Staircase_std.cxx
 *  @brief The test file of class Staircase for standard methods
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2007-09-14 10:44:28 +0200 (ven 14 sep 2007) $
 *  Id:      $Id: t_Staircase_std.cxx 516 2007-09-14 08:44:28Z dutka $
 */
#include <iostream>
#include <sstream>
#include <cstdlib>

#include "OT.hxx"
#include "OTtestcode.hxx"
#include "Normal.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"
#include "CorrelationMatrix.hxx"
#include "IdentityMatrix.hxx"
#include "Graph.hxx"
#include "Staircase.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Stat;
using namespace OT::Base::Graph;
using namespace OT::Uncertainty::Distribution;
using namespace std;


int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  setRandomGenerator();

  try {

    // Instanciate one distribution object
    UnsignedLong dim(1);
    NumericalPoint meanPoint(dim, 1.0);
    meanPoint[0] = 0.5;
    NumericalPoint sigma(dim, 1.0);
    sigma[0] = 2.0;
    CorrelationMatrix R = IdentityMatrix(dim);
    Normal distribution1(meanPoint, sigma, R);

    // Instanciate another distribution object
    meanPoint[0] = -1.5;
    sigma[0] = 4.0;
    Normal distribution2(meanPoint, sigma, R);

    // Test for sampling
    UnsignedLong size = 2000;
    UnsignedLong nPoints = 20;
    NumericalSample sample1(distribution1.getNumericalSample( size ));
    NumericalSample sample2(distribution2.getNumericalSample( size ));
    
    // Construct empirical CDF for each sample
    NumericalSample data1(nPoints, 2), data2(nPoints, 2);
    NumericalPoint cursor1(2);
    NumericalPoint cursor2(2);
    NumericalScalar count1;
    NumericalScalar count2;

    for(UnsignedLong i = 0; i < nPoints; i++)
      {
	cursor1[0] = 13.*i/nPoints - 6.5;
	count1 = 0;
	cursor2[0] = 24.*i/nPoints - 13.5;
	count2 = 0;
	
	for(UnsignedLong j = 0; j < size; j++)
	  {
	    if(sample1[j][0] < cursor1[0]) count1++;
	    if(sample2[j][0] < cursor2[0]) count2++;
	  }

	cursor1[1] = count1/size;
	cursor2[1] = count2/size;

	data1[i] = cursor1;
	data2[i] = cursor2;
      }

    // Create an empty graph
    Graph myGraph("Some curves", "x1", "x2", true,"bottomright");

    // Create the first staircase
    Staircase myStaircase1(data1, "blue", "solid", "s", "");

    Staircase myStaircase1b(myStaircase1);
    myStaircase1b.setPattern("S");
    myStaircase1b.setColor("green");
    myStaircase1b.setLineStyle("dashed");
    myStaircase1b.setLegendName("eCDF1b, pat=S");
    
    // Then, draw it
    myGraph.addDrawable(myStaircase1);
    myGraph.addDrawable(myStaircase1b);
    myGraph.draw("Graph_Staircase_a_OT", 640, 480);

    // Check that the correct files have been generated by computing their checksum
    std::cout << "bitmap=" << myGraph.getBitmap() << std::endl;
    std::cout << "postscript=" << myGraph.getPostscript() << std::endl;

    // Create the second staircase
    Staircase myStaircase2(data2, "red", "dashed","S","eCDF2, pat=S");

    // Add it to the graph and draw everything
    myGraph.addDrawable(myStaircase2);
    myGraph.draw("Graph_Staircase_b_OT", 640, 480);
    std::cout << "bitmap=" << myGraph.getBitmap() << std::endl;
    std::cout << "postscript=" << myGraph.getPostscript() << std::endl;
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
