//                                               -*- C++ -*-
/**
 *  @file  t_KernelMixture_std.cxx
 *  @brief The test file of class KernelMixture for standard methods
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu 26 jun 2008) $
 *  Id:      $Id: t_KernelMixture_std.cxx 862 2008-06-26 11:50:17Z dutka $
 */
#include <iostream>
#include <sstream>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "KernelMixture.hxx"
#include "Mixture.hxx"
#include "Normal.hxx"
#include "IdentityMatrix.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Stat;
using namespace OT::Uncertainty::Model;
using namespace OT::Uncertainty::Distribution;




int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  setRandomGenerator();

  try {

    UnsignedLong dimension(3);
    NumericalPoint meanPoint(dimension, 1.0);
    meanPoint[0] = 0.5;
    meanPoint[1] = -0.5;
    NumericalPoint sigma(dimension, 1.0);
    sigma[0] = 2.0;
    sigma[1] = 3.0;

    NumericalSample sample(0, dimension);
    // Create a collection of distribution
    Mixture::DistributionCollection aCollection;

    aCollection.add( Normal(meanPoint, sigma, IdentityMatrix(dimension)) );
    sample.add(meanPoint);
    meanPoint += NumericalPoint(dimension, 1.0);
    aCollection.add( Normal(meanPoint, sigma, IdentityMatrix(dimension)) );
    sample.add(meanPoint);
    meanPoint += NumericalPoint(dimension, 1.0);
    aCollection.add( Normal(meanPoint, sigma, IdentityMatrix(dimension)) );
    sample.add(meanPoint);

    // Instanciate one distribution object
    Normal kernel;
    KernelMixture distribution(kernel, sigma, sample);
    std::cout << "Distribution " << distribution << std::endl;
    Mixture distributionRef(aCollection);

    // Is this distribution elliptical ?
    std::cout << "Elliptical = " << (distribution.isElliptical() ? "true" : "false") << std::endl;

    // Is this distribution continuous ?
    std::cout << "Continuous = " << (distribution.isContinuous() ? "true" : "false") << std::endl;

    // Test for realization of distribution
    NumericalPoint oneRealization = distribution.getRealization();
    std::cout << "oneRealization=" << oneRealization << std::endl;

    // Test for sampling
    UnsignedLong size = 10000;
    NumericalSample oneSample = distribution.getNumericalSample( size );
    std::cout << "oneSample first=" << oneSample[0] << " last=" << oneSample[size - 1] << std::endl;
    std::cout << "mean=" << oneSample.computeMean() << std::endl;
    std::cout << "covariance=" << oneSample.computeCovariance() << std::endl;

    // Define a point
    NumericalPoint point( dimension, 1.0 );
    std::cout << "Point= " << point << std::endl;

    // Show PDF and CDF of point
    NumericalScalar eps(1e-5);
    NumericalPoint DDF = distribution.computeDDF( point );
    std::cout << "ddf     =" << DDF << std::endl;
    std::cout << "ddf (ref)=" << distributionRef.computeDDF( point ) << std::endl;
    NumericalPoint ddfFD(dimension);
    for (UnsignedLong i = 0; i < dimension; i++)
      {
	NumericalPoint left(point);
	left[i] += eps;
	NumericalPoint right(point);
	right[i] -= eps;
	ddfFD[i] = (distribution.computePDF(left) - distribution.computePDF(right)) / (2.0 * eps);
      }
    std::cout << "ddf (FD)=" << ddfFD << std::endl;
    NumericalScalar PDF = distribution.computePDF( point );
    std::cout << "pdf     =" << PDF << std::endl;
    std::cout << "pdf (ref)=" << distributionRef.computePDF( point ) << std::endl;
    if (dimension == 1)
      {
	std::cout << "pdf (FD)=" << (distribution.computeCDF( point + NumericalPoint(1, eps) ) - distribution.computeCDF( point  + NumericalPoint(1, -eps) )) / (2.0 * eps) << std::endl;
      }
    NumericalScalar CDF = distribution.computeCDF( point );
    std::cout << "cdf=" << CDF << std::endl;
    std::cout << "cdf (ref)=" << distributionRef.computeCDF( point ) << std::endl;
    NumericalPoint quantile = distribution.computeQuantile( 0.95 );
    std::cout << "quantile=" << quantile << std::endl;
    std::cout << "quantile (ref)=" << distributionRef.computeQuantile( 0.95 ) << std::endl;
    std::cout << "cdf(quantile)=" << distribution.computeCDF(quantile) << std::endl;
    NumericalPoint mean = distribution.getMean();
    std::cout << "mean=" << mean << std::endl;
    std::cout << "mean (ref)=" << distributionRef.getMean() << std::endl;
    CovarianceMatrix covariance = distribution.getCovariance();
    std::cout << "covariance=" << covariance << std::endl;
    std::cout << "covariance (ref)=" << distributionRef.getCovariance() << std::endl;
    //    KernelMixture::NumericalPointCollection parameters = distribution.getParametersCollection();
    //    std::cout << "parameters=" << parameters << std::endl;
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
