//                                               -*- C++ -*-
/**
 *  @file  t_GumbelCopula_std.cxx
 *  @brief The test file of class GumbelCopula for standard methods
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu 26 jun 2008) $
 *  Id:      $Id: t_GumbelCopula_std.cxx 862 2008-06-26 11:50:17Z dutka $
 */
#include <iostream>
#include <sstream>
#include <cmath>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "GumbelCopula.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"
#include "Copula.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Stat;
using namespace OT::Uncertainty::Model;
using namespace OT::Uncertainty::Distribution;

int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  setRandomGenerator();

  try {
    // Instanciate one distribution object
    UnsignedLong dim(2);
    GumbelCopula copula(2.5);
    copula.setName("a gumbel copula");
    std::cout << "Copula " << copula << std::endl;
    std::cout << "Mean " << copula.getMean() << std::endl;
    std::cout << "Covariance " << copula.getCovariance() << std::endl;
    // Is this copula an elliptical distribution?
    std::cout << "Elliptical distribution= " << (copula.isElliptical() ? "true" : "false") << std::endl;

    // Is this copula elliptical ?
    std::cout << "Elliptical copula= " << (copula.hasEllipticalCopula() ? "true" : "false") << std::endl;

    // Is this copula independent ?
    std::cout << "Independent copula= " << (copula.hasIndependentCopula() ? "true" : "false") << std::endl;

    // Test for realization of copula
    NumericalPoint oneRealization = copula.getRealization();
    std::cout << "oneRealization=" << oneRealization << std::endl;

    // Test for sampling
    UnsignedLong size = 10;
    NumericalSample oneSample = copula.getNumericalSample( size );
    std::cout << "oneSample=" << oneSample << std::endl;

    // Test for sampling
    size = 10000;
    NumericalSample anotherSample = copula.getNumericalSample( size );
    std::cout << "anotherSample mean=" << anotherSample.computeMean() << std::endl;
    std::cout << "anotherSample covariance=" << anotherSample.computeCovariance() << std::endl;

    // Define a point
    NumericalPoint point(dim, 0.2);

    // Show PDF and CDF of zero point
    NumericalScalar zeroPDF = copula.computePDF( point );
    NumericalScalar zeroCDF = copula.computeCDF( point );
    std::cout << "point= " << point
	      << " pdf=" << zeroPDF
	      << " cdf=" << zeroCDF
	      << std::endl;
    // Get 50% quantile
    NumericalPoint quantile = copula.computeQuantile( 0.5 );
    std::cout << "Quantile=" << quantile << std::endl;
    std::cout << "CDF(quantile)=" << copula.computeCDF(quantile) << std::endl;
    // Extract the marginals
    for (UnsignedLong i = 0; i < dim; i++)
      {
	Copula margin(copula.getMarginal(i));
    	std::cout << "margin=" << margin << std::endl;
    	std::cout << "margin PDF=" << margin.computePDF(NumericalPoint(1,0.25)) << std::endl;
    	std::cout << "margin CDF=" << margin.computeCDF(NumericalPoint(1,0.25)) << std::endl;
    	std::cout << "margin quantile=" << margin.computeQuantile(0.95) << std::endl;
    	std::cout << "margin realization=" << margin.getRealization() << std::endl;
      }

    // Extract a 2-D marginal
    GumbelCopula::Indices indices(2, 0);
    indices[0] = 1;
    indices[1] = 0;
    std::cout << "indices=" << indices << std::endl;
    Copula margins(copula.getMarginal(indices));
    std::cout << "margins=" << margins << std::endl;
    std::cout << "margins PDF=" << margins.computePDF(NumericalPoint(2, 0.25)) << std::endl;
    std::cout << "margins CDF="<< margins.computeCDF(NumericalPoint(2, 0.25)) << std::endl;
    quantile = margins.computeQuantile(0.95);
    std::cout << "margins quantile=" << quantile << std::endl;
    std::cout << "margins CDF(quantile)=" << margins.computeCDF(quantile) << std::endl;
    std::cout << "margins realization=" << margins.getRealization() << std::endl;
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
