//                                               -*- C++ -*-
/**
 *  @file  t_ComposedDistribution_std.cxx
 *  @brief The test file of class ComposedDistribution for standard methods
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-13 22:37:56 +0200 (sam 13 sep 2008) $
 *  Id:      $Id: t_ComposedDistribution_std.cxx 929 2008-09-13 20:37:56Z dutka $
 */
#include <iostream>
#include <sstream>
#include "OT.hxx"
#include "OTtestcode.hxx"
#include "ComposedDistribution.hxx"
#include "Student.hxx"
#include "Normal.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"
#include "IndependentCopula.hxx"
#include "NormalCopula.hxx"
#include "IdentityMatrix.hxx"
#include "Indices.hxx"
#include "PlatformInfo.hxx"

using namespace OT;
using namespace OT::Test;
using namespace OT::Base::Common;
using namespace OT::Base::Type;
using namespace OT::Base::Stat;
using namespace OT::Uncertainty::Model;
using namespace OT::Uncertainty::Distribution;




int main(int argc, char *argv[])
{
  TESTPREAMBLE;
  setRandomGenerator();

  try {

    // Test basic functionnalities
    //checkClassWithClassName<TestObject>();

    // Test some extra functionnalities
    //checkNameFeature<TestObject>();
    NumericalPoint mean(3);
    mean[0] = 1.0;
    mean[1] = 2.0;
    mean[2] = 3.0;
    NumericalPoint sigma(3);
    sigma[0] = 2.0;
    sigma[1] = 3.0;
    sigma[2] = 4.0;
    // Create a collection of distribution
    ComposedDistribution::DistributionCollection aCollection;
    Normal marginal(mean[0], sigma[0]);
    marginal.setName("First");
    Description component(1);
    component[0] = "One";
    marginal.setDescription(component);
    aCollection.add( Distribution(marginal, "First") );
    marginal = Normal(mean[1], sigma[1]);
    marginal.setName("Second");
    component[0] = "Two";
    marginal.setDescription(component);
    aCollection.add( Distribution(marginal, "Second") );
    marginal = Normal(mean[2], sigma[2]);
    marginal.setName("Third");
    component[0] = "Three";
    marginal.setDescription(component);
    aCollection.add( Distribution(marginal, "Third") );

    // Create a copula
    IndependentCopula aCopula(aCollection.getSize());
    aCopula.setName("Independent copula");
    // Instanciate one distribution object
    ComposedDistribution distribution(aCollection, Copula(aCopula));
    distribution.setName("myDist");
    UnsignedLong dim(distribution.getDimension());
    std::cout << "Distribution " << distribution << std::endl;
    std::cout << "Parameters " << distribution.getParametersCollection() << std::endl;
    std::cout << "Mean " << distribution.getMean() << std::endl;
    UnsignedLong precision(PlatformInfo::GetNumericalPrecision());
    PlatformInfo::SetNumericalPrecision(2);
    std::cout << "Covariance " << distribution.getCovariance() << std::endl;
    PlatformInfo::SetNumericalPrecision(precision);
    // Is this distribution an elliptical distribution?
    std::cout << "Elliptical distribution= " << (distribution.isElliptical() ? "true" : "false") << std::endl;

    // Has this distribution an elliptical copula?
    std::cout << "Elliptical copula= " << (distribution.hasEllipticalCopula() ? "true" : "false") << std::endl;

    // Has this distribution an independent copula?
    std::cout << "Independent copula= " << (distribution.hasIndependentCopula() ? "true" : "false") << std::endl;

    // Test for realization of distribution
    NumericalPoint oneRealization = distribution.getRealization();
    std::cout << "oneRealization=" << oneRealization << std::endl;

    // Test for sampling
    UnsignedLong size = 10;
    NumericalSample oneSample = distribution.getNumericalSample( size );
    std::cout << "oneSample=" << oneSample << std::endl;

    // Test for sampling
    size = 10000;
    NumericalSample anotherSample = distribution.getNumericalSample( size );
    std::cout << "anotherSample mean=" << anotherSample.computeMean() << std::endl;
    std::cout << "anotherSample covariance=" << anotherSample.computeCovariance() << std::endl;

    // Define a point
    NumericalPoint zero(dim, 0.0);

    // Show PDF and CDF of zero point
    NumericalScalar zeroPDF = distribution.computePDF( zero );
    NumericalScalar zeroCDF = distribution.computeCDF( zero );
    std::cout << "Zero point= " << zero 
	      << " pdf=" << zeroPDF
	      << " cdf=" << zeroCDF
	      << std::endl;
    // Get 95% quantile
    NumericalPoint quantile = distribution.computeQuantile( 0.95 );
    std::cout << "Quantile=" << quantile << std::endl;
    std::cout << "CDF(quantile)=" << distribution.computeCDF(quantile) << std::endl;
    // Reference
    Normal ref(mean, sigma, IdentityMatrix(distribution.getDimension()));
    std::cout << "Reference=" << std::endl;
    std::cout << "Zero point= " << zero
	      << " pdf= " << ref.computePDF(zero)
	      << " cdf= " << ref.computeCDF(zero)
	      << " quantile= " << ref.computeQuantile(0.95)
	      << std::endl;
    // Extract the marginals
    for (UnsignedLong i = 0; i < dim; i++)
      {
	Distribution margin(distribution.getMarginal(i));
    	std::cout << "margin=" << margin << std::endl;
    	std::cout << "margin PDF=" << margin.computePDF(NumericalPoint(1)) << std::endl;
    	std::cout << "margin CDF=" << margin.computeCDF(NumericalPoint(1)) << std::endl;
    	std::cout << "margin quantile=" << margin.computeQuantile(0.95) << std::endl;
    	std::cout << "margin realization=" << margin.getRealization() << std::endl;
      }

    // Extract a 2-D marginal
    Indices indices(2, 0);
    indices[0] = 1;
    indices[1] = 0;
    std::cout << "indices=" << indices << std::endl;
    Distribution margins(distribution.getMarginal(indices));
    std::cout << "margins=" << margins << std::endl;
    std::cout << "margins PDF=" << margins.computePDF(NumericalPoint(2)) << std::endl;
    std::cout << "margins CDF="<< margins.computeCDF(NumericalPoint(2)) << std::endl;
    quantile = margins.computeQuantile(0.5);
    std::cout << "margins quantile=" << quantile << std::endl;
    std::cout << "margins CDF(quantile)=" << margins.computeCDF(quantile) << std::endl;
    std::cout << "margins realization=" << margins.getRealization() << std::endl;
    // With a Normal copula
    CorrelationMatrix correlation(dim);
    for(UnsignedLong i = 1; i < dim; i++)
      {
	correlation(i - 1, i) = 0.25;
      }
    NormalCopula anotherCopula(correlation);
    anotherCopula.setName("Normal copula");
    distribution = ComposedDistribution(aCollection, Copula(anotherCopula));
    distribution.setName("myDist");
    Normal distributionRef(mean, sigma, correlation);
    std::cout << "Distribution " << distribution << std::endl;
    std::cout << "Parameters " << distribution.getParametersCollection() << std::endl;
    // Show PDF and CDF at point
    NumericalPoint point(dim, 0.0);
    std::cout << "PDF      =" << distribution.computePDF(point) << std::endl;
    std::cout << "PDF (ref)=" << distributionRef.computePDF(point) << std::endl;
    std::cout << "CDF      =" << distribution.computeCDF(point) << std::endl;
    std::cout << "CDF (ref)=" << distributionRef.computeCDF(point) << std::endl;
    // 95% quantile
    quantile = distribution.computeQuantile( 0.95 );
    std::cout << "Quantile      =" << quantile << std::endl;
    std::cout << "Quantile (ref)=" << distributionRef.computeQuantile( 0.95 ) << std::endl;
    std::cout << "CDF(quantile)=" << distribution.computeCDF(quantile) << std::endl;
    std::cout << "Mean      =" << distribution.getMean() << std::endl;
    std::cout << "Mean (ref)=" << distributionRef.getMean() << std::endl;
    NumericalPoint standardDeviation = distribution.getStandardDeviation();
    std::cout << "Standard deviation      =" << standardDeviation << std::endl;
    std::cout << "Standard deviation (ref)=" << distributionRef.getStandardDeviation() << std::endl;
    NumericalPoint skewness = distribution.getSkewness();
    std::cout << "Skewness       =" << skewness << std::endl;
    std::cout << "Skewness (ref)==" << distributionRef.getSkewness() << std::endl;
    NumericalPoint kurtosis = distribution.getKurtosis();
    std::cout << "Kurtosis      =" << kurtosis << std::endl;
    std::cout << "Kurtosis (ref)=" << distributionRef.getKurtosis() << std::endl;
    std::cout << "Covariance      =" << distribution.getCovariance() << std::endl;
    std::cout << "Covariance (ref)=" << distributionRef.getCovariance() << std::endl;
    anotherSample = distribution.getNumericalSample(size);
    std::cout << "anotherSample mean=" << anotherSample.computeMean() << std::endl;
    std::cout << "anotherSample covariance=" << anotherSample.computeCovariance() << std::endl;
  }
  catch (TestFailed & ex) {
    std::cerr << ex << std::endl;
    return ExitCode::Error;
  }


  return ExitCode::Success;
}
