//                                               -*- C++ -*-
/**
 *  @file  Uniform.cxx
 *  @brief The Uniform distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: Uniform.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include <cmath>
#include "Uniform.hxx"
#include "RandomGenerator.hxx"
#include "IdentityMatrix.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Type::IdentityMatrix  IdentityMatrix;
      typedef Base::Stat::RandomGenerator RandomGenerator;

      CLASSNAMEINIT(Uniform);

      static Base::Common::Factory<Uniform> RegisteredFactory("Uniform");

      /* Default constructor */
      Uniform::Uniform()
	: EllipticalDistribution(NumericalPoint(1, 0.0),
					       // 0.2886751345948128822545744 = 1 / sqrt(12)
					       NumericalPoint(1, 2.0 * 0.2886751345948128822545744),
					       IdentityMatrix(1),
					       "Uniform"),
	  a_(-1.0), b_(1.0)
      {
	setDimension( 1 );
	computeRange();
      }

      /* Parameters constructor */
      Uniform::Uniform(const NumericalScalar a,
		       const NumericalScalar b)
	: EllipticalDistribution(NumericalPoint(1, 0.5 * (a + b)),
					       // 0.2886751345948128822545744 = 1 / sqrt(12)
					       NumericalPoint(1, (b - a) * 0.2886751345948128822545744),
					       IdentityMatrix(1),
					       "Uniform"),
	  a_(a), b_(b)
      {
	if (b_ <= a_) throw InvalidArgumentException(HERE) << "Error the lower bound a of a Uniform distribution must be lesser than its upper bound b, here a=" << a << " b=" << b;
	setDimension( 1 );
	computeRange();
      }

      /* Comparison operator */
      Bool Uniform::operator ==(const Uniform & other) const {
	Bool sameObject = false;

	if (this != &other) { // Other is NOT me, so I have to realize the comparison
	  // sameObject = ...
	  // TODO: Write Uniform::operator ==(...)
	  sameObject = (a_ == other.a_) && (b_ == other.b_);
	} else sameObject = true;

	return sameObject;
      }
  
      /* String converter */
      String Uniform::str() const {
	OSS oss;
	oss << "class=" << Uniform::GetClassName()
	    << " name=" << getName()
	    << " dimension=" << getDimension()
	    << " a=" << a_
	    << " b=" << b_;
	return oss;
      }
  
      /* Virtual constructor */
      Uniform * Uniform::clone() const
      {
	return new Uniform(*this);
      }

      /* Compute the numerical range of the distribution given the parameters values */
      void Uniform::computeRange()
      {
	setRange(Interval(a_, b_));
      }


      /* Get one realization of the distribution */
      Uniform::NumericalPoint Uniform::getRealization() const
      {
	return NumericalPoint(1, a_ + (b_ - a_) * RandomGenerator::Generate());
      }     


      /* Get the DDF of the distribution */
      Uniform::NumericalPoint Uniform::computeDDF(const NumericalPoint & point) const
      {
        return NumericalPoint(1, 0.0);
      }


      /* Get the PDF of the distribution */
      NumericalScalar Uniform::computePDF(const NumericalPoint & point) const
      {
	const NumericalScalar x(point[0]);
	if ((x <= a_) || (x > b_)) return 0.0;
	return 1.0 / (b_ - a_);
      }


      /* Get the CDF of the distribution */
      NumericalScalar Uniform::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
	const NumericalScalar x(point[0]);
	if (x <= a_) return (tail ? 1.0 : 0.0);
	if (x > b_) return (tail ? 0.0 : 1.0);
	if (tail) return (b_ - x) / (b_ - a_);
	return (x - a_) / (b_ - a_);
      }

      /* Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
      NumericalComplex Uniform::computeCharacteristicFunction(const NumericalPoint & point) const
      {
	const NumericalScalar u(point[0]);
	const NumericalScalar au(a_ * u);
	const NumericalScalar bu(b_ * u);
	if (fabs(u) <= 1.0e-10) return NumericalComplex(1.0, 0.5 * (au + bu));
	const NumericalScalar idenom(1.0 / (bu - au));
	return NumericalComplex(idenom * (sin(bu) - sin(au)), idenom * (cos(au) - cos(bu)));
      }

      /* Get the PDFGradient of the distribution */
      Uniform::NumericalPoint Uniform::computePDFGradient(const NumericalPoint & point) const
      {
	const NumericalScalar x(point[0]);
	NumericalPoint pdfGradient(2, 0.0);
	if ((x <= a_) || (x > b_)) return pdfGradient;
	const NumericalScalar iAB(1.0 / (b_ - a_));
	const NumericalScalar iAB2(iAB * iAB);
	pdfGradient[0] = iAB2;
	pdfGradient[1] = -iAB2;
	return pdfGradient;
      }

      /* Get the CDFGradient of the distribution */
      Uniform::NumericalPoint Uniform::computeCDFGradient(const NumericalPoint & point) const
      {
	const NumericalScalar x(point[0]);
	NumericalPoint cdfGradient(2, 0.0);
	if ((x <= a_) || (x > b_)) return cdfGradient;
	const NumericalScalar iAB(1.0 / (b_ - a_));
	const NumericalScalar iAB2(iAB * iAB);
	cdfGradient[0] = (x - b_) * iAB2;
	cdfGradient[1] = (a_ - x) * iAB2;
	return cdfGradient;
      }

      /* Get the quantile of the distribution */
      Uniform::NumericalPoint Uniform::computeQuantile(const NumericalScalar prob) const
      {
	if (prob < 0.0 || prob > 1.0) throw InvalidArgumentException(HERE) << "Error: cannot compute a quantile for a probability level outside of [0, 1]";
        // Special case for boarding values
        if (prob == 0.0) return getRange().getLowerBound();
        if (prob == 1.0) return getRange().getUpperBound();
	return NumericalPoint(1, a_ + prob * (b_ - a_));
      }

      /* Get the roughness, i.e. the L2-norm of the PDF */
      NumericalScalar Uniform::getRoughness() const
      {
	return 1.0 / (b_ - a_);
      }

      /* Get the skewness of the distribution */
      Uniform::NumericalPoint Uniform::getSkewness() const throw(NotDefinedException)
      {
	return NumericalPoint(1, 0.0);
      }

      /* Get the kurtosis of the distribution */
      Uniform::NumericalPoint Uniform::getKurtosis() const throw(NotDefinedException)
      {
	// 1.8 = 9/5
	return NumericalPoint(1, 1.8);
      }

      /* Compute the density generator of the ellipticalal generator, i.e.
       *  the function phi such that the density of the distribution can
       *  be written as p(x) = phi(t(x-mu)S^(-1)(x-mu))                      */
      NumericalScalar Uniform::computeDensityGenerator(const NumericalScalar betaSquare) const
      {
	if (betaSquare > 3.0) return 0.0;
	return 1.0 / (b_ - a_);
      }
      
      /* Compute the derivative of the density generator */
      NumericalScalar Uniform::computeDensityGeneratorDerivative(const NumericalScalar betaSquare) const
      {
	return 0.0;
      }
      
      /* Compute the second derivative of the density generator */
      NumericalScalar Uniform::computeDensityGeneratorSecondDerivative(const NumericalScalar betaSquare) const
      {
	return 0.0;
      }
      

      /* Parameters value and description accessor */
      Uniform::NumericalPointWithDescriptionCollection Uniform::getParametersCollection() const
      {
	NumericalPointWithDescriptionCollection parameters(1);
	NumericalPointWithDescription point(2);
        Description description(point.getDimension());
	point[0] = a_;
	point[1] = b_;
	description[0] = "a";
	description[1] = "b";
	point.setDescription(description);
	point.setName(getDescription()[0]);
	parameters[0] = point;
	return parameters;
      }



      /* A accessor */
      void Uniform::setA(const NumericalScalar a)
      {
	a_ = a;
        setMean(NumericalPoint(1, 0.5 * (a_ + b_)));
	// 0.2886751345948128822545744 = 1 / sqrt(12)
        setSigma(NumericalPoint(1, (b_ - a_) * 0.2886751345948128822545744));
	computeRange();
      }

      NumericalScalar Uniform::getA() const
      {
	return a_;
      }


      /* B accessor */
      void Uniform::setB(const NumericalScalar b)
      {
	b_ = b;
        setMean(NumericalPoint(1, 0.5 * (a_ + b_)));
	// 0.2886751345948128822545744 = 1 / sqrt(12)
        setSigma(NumericalPoint(1, (b_ - a_) * 0.2886751345948128822545744));
	computeRange();
      }

      NumericalScalar Uniform::getB() const
      {
	return b_;
      }

      /* Quantile computation for dimension=1 */
      NumericalScalar Uniform::computeScalarQuantile(const NumericalScalar prob, const NumericalScalar initialGuess, const NumericalScalar initialStep) const
      {
	if (prob < 0.0 || prob > 1.0) throw InvalidArgumentException(HERE) << "Error: cannot compute a quantile for a probability level outside of [0, 1]";
        // Special case for boarding values
        if (prob == 0.0) return a_;
        if (prob == 1.0) return b_;
	return a_ + prob * (b_ - a_);
      }

      /* Method save() stores the object through the StorageManager */
      void Uniform::save(const StorageManager::Advocate & adv) const
      {
	EllipticalDistribution::save(adv);
	adv.writeValue("a_", a_);
	adv.writeValue("b_", b_);
      }

      /* Method load() reloads the object from the StorageManager */
      void Uniform::load(const StorageManager::Advocate & adv)
      {
	EllipticalDistribution::load(adv);

	String name;
	NumericalScalar value;
	StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, value)) {
	    if (name == "a_") a_ = value;
	    if (name == "b_") b_ = value;
	  }
	}
	computeRange();
      }
      
    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
