//                                               -*- C++ -*-
/**
 *  @file  TruncatedDistribution.hxx
 *  @brief The TruncatedDistribution distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-13 22:37:56 +0200 (sam 13 sep 2008) $
 *  Id:      $Id: TruncatedDistribution.hxx 929 2008-09-13 20:37:56Z dutka $
 */
#ifndef OPENTURNS_TRUNCATEDDISTRIBUTION_HXX
#define OPENTURNS_TRUNCATEDDISTRIBUTION_HXX

#include "NonEllipticalDistribution.hxx"
#include "Distribution.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class TruncatedDistribution
       *
       * The TruncatedDistribution distribution.
       */
      class TruncatedDistribution 
	: public Model::NonEllipticalDistribution
      {
	CLASSNAME;
      public:

	typedef Base::Common::InvalidArgumentException               InvalidArgumentException;
	typedef Model::NonEllipticalDistribution                     NonEllipticalDistribution;  // required by SWIG
	typedef Model::Distribution                                  Distribution;
	typedef NonEllipticalDistribution::NumericalPoint            NumericalPoint;
	typedef NonEllipticalDistribution::NumericalSample           NumericalSample;
	typedef NonEllipticalDistribution::CovarianceMatrix          CovarianceMatrix;
	typedef NonEllipticalDistribution::NumericalPointWithDescriptionCollection  NumericalPointWithDescriptionCollection;
	typedef NonEllipticalDistribution::NotDefinedException       NotDefinedException;
	typedef NonEllipticalDistribution::StorageManager            StorageManager;

	// If the mass of the underlying distribution is at least this value, use rejection for sampling
	static const NumericalScalar DefaultThresholdRealization = 0.5;

	enum BoundSide { LOWER, UPPER };

	/** Default constructor */
	TruncatedDistribution();

	/** Parameters constructor to use when the two bounds are finite */
	TruncatedDistribution(const Distribution & distribution,
			      const NumericalScalar lowerBound,
			      const NumericalScalar upperBound,
			      const NumericalScalar thresholdRealization = DefaultThresholdRealization)
	  throw (InvalidArgumentException);

	/** Parameters constructor to use when one of the bounds is not finite */
	TruncatedDistribution(const Distribution & distribution,
			      const NumericalScalar bound,
			      const BoundSide side = LOWER,
			      const NumericalScalar thresholdRealization = DefaultThresholdRealization)
	  throw (InvalidArgumentException);

	/** Parameters constructor to use when the two bounds are finite */
	TruncatedDistribution(const Distribution & distribution,
			      const Interval & truncationInterval,
			      const NumericalScalar thresholdRealization = DefaultThresholdRealization)
	  throw (InvalidArgumentException);

	/** Comparison operator */
	Bool operator ==(const TruncatedDistribution & other) const;

	/** String converter */
	String str() const;

	/* Interface inherited from Distribution */

	/** Virtual constructor */
	virtual TruncatedDistribution * clone() const;
	
	/** Get one realization of the distribution */
	virtual NumericalPoint getRealization() const;

	/** Get the DDF of the distribution */
	using NonEllipticalDistribution::computeDDF;
	virtual NumericalPoint computeDDF(const NumericalPoint & point) const;

	/** Get the PDF of the distribution */
	using NonEllipticalDistribution::computePDF;
	virtual NumericalScalar computePDF(const NumericalPoint & point) const;

	/** Get the CDF of the distribution */
	using NonEllipticalDistribution::computeCDF;
	virtual NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

	/** Get the PDFGradient of the distribution */
	virtual NumericalPoint computePDFGradient(const NumericalPoint & point) const;

	/** Get the CDFGradient of the distribution */
	virtual NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

	/** Get the quantile of the distribution */
	virtual NumericalPoint computeQuantile(const NumericalScalar prob) const;

	/** Parameters value and description accessor */
	virtual NumericalPointWithDescriptionCollection getParametersCollection() const;


	/* Interface specific to TruncatedDistribution */

	/** Distribution accessor */
	void setDistribution(const Distribution & distribution) throw (InvalidArgumentException);
	Distribution getDistribution() const;

	/** Lower bound accessor */
	void setLowerBound(const NumericalScalar lowerBound) throw (InvalidArgumentException);
	NumericalScalar getLowerBound() const;

	/** Lower bound finite flag accessor */
	void setFiniteLowerBound(const Bool finiteLowerBound) throw (InvalidArgumentException);
        Bool getFiniteLowerBound() const;

	/** Upper bound accessor */
	void setUpperBound(const NumericalScalar upperBound) throw (InvalidArgumentException);
	NumericalScalar getUpperBound() const;

	/** Upper bound finite flag accessor */
	void setFiniteUpperBound(const Bool finiteUpperBound) throw (InvalidArgumentException);
	Bool getFiniteUpperBound() const;

	/** Threshold realization accessor */
	void setThresholdRealization(const NumericalScalar thresholdRealization)
	  throw (InvalidArgumentException);
	NumericalScalar getThresholdRealization() const;

	/** Method save() stores the object through the StorageManager */
	virtual void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	virtual void load(const StorageManager::Advocate & adv);


      protected:
    
    
      private:

	/** Compute the numerical range of the distribution given the parameters values */
	void computeRange();

	/** The main parameter set of the distribution */
	Distribution distribution_;
	NumericalScalar lowerBound_;
	Bool finiteLowerBound_;
	NumericalScalar upperBound_;
	Bool finiteUpperBound_;
	NumericalScalar thresholdRealization_;
	/** Usefull quantities */
	NumericalScalar pdfLowerBound_;
	NumericalScalar pdfUpperBound_;
	NumericalScalar cdfLowerBound_;
	NumericalScalar cdfUpperBound_;
	NumericalScalar normalizationFactor_;
      }; /* class TruncatedDistribution */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_TRUNCATEDDISTRIBUTION_HXX */
