//                                               -*- C++ -*-
/**
 *  @file  Triangular.hxx
 *  @brief The Triangular distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-13 22:37:56 +0200 (sam 13 sep 2008) $
 *  Id:      $Id: Triangular.hxx 929 2008-09-13 20:37:56Z dutka $
 */
#ifndef OPENTURNS_TRIANGULAR_HXX
#define OPENTURNS_TRIANGULAR_HXX

#include "OT.hxx"
#include "NonEllipticalDistribution.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class Triangular
       *
       * The Triangular distribution.
       */
      class Triangular 
	: public Model::NonEllipticalDistribution
      {
	CLASSNAME;
      public:

	typedef Base::Common::InvalidArgumentException               InvalidArgumentException;
	typedef Model::NonEllipticalDistribution                     NonEllipticalDistribution;  // required by SWIG
	typedef NonEllipticalDistribution::NumericalPoint            NumericalPoint;
	typedef NonEllipticalDistribution::NumericalSample           NumericalSample;
	typedef NonEllipticalDistribution::CovarianceMatrix          CovarianceMatrix;
	typedef NonEllipticalDistribution::NumericalPointWithDescriptionCollection  NumericalPointWithDescriptionCollection;
	typedef NonEllipticalDistribution::NotDefinedException       NotDefinedException;
	typedef NonEllipticalDistribution::StorageManager            StorageManager;

	/** Default constructor */
        Triangular();

	/** Parameters constructor */
        Triangular(const NumericalScalar a,
		   const NumericalScalar m,
		   const NumericalScalar b)
	  throw (InvalidArgumentException);


	/** Comparison operator */
	Bool operator ==(const Triangular & other) const;

	/** String converter */
	String str() const;



	/* Interface inherited from Distribution */

	/** Virtual constructor */
	Triangular * clone() const;

	/** Get one realization of the Triangular distribution */
	NumericalPoint getRealization() const;

	/** Get the DDF of the Triangular distribution */
	using NonEllipticalDistribution::computeDDF;
	NumericalPoint computeDDF(const NumericalPoint & point) const;

	/** Get the PDF of the Triangular distribution */
	using NonEllipticalDistribution::computePDF;
	NumericalScalar computePDF(const NumericalPoint & point) const;

	/** Get the CDF of the Triangular distribution */
	using NonEllipticalDistribution::computeCDF;
	NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

	/** Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
	NumericalComplex computeCharacteristicFunction(const NumericalPoint & point) const;

	/** Get the PDFGradient of the Triangular distribution */
	NumericalPoint computePDFGradient(const NumericalPoint & point) const;

	/** Get the CDFGradient of the Triangular distribution */
	NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

	/** Get the quantile of the Triangular distribution */
	NumericalPoint computeQuantile(const NumericalScalar prob) const;
	
	/** Get the roughness, i.e. the L2-norm of the PDF */
	NumericalScalar getRoughness() const;

	/** Get the mean of the distribution */
	NumericalPoint getMean() const throw(NotDefinedException);

	/** Get the standard deviation of the distribution */
	NumericalPoint getStandardDeviation() const throw(NotDefinedException);

	/** Get the skewness of the distribution */
	NumericalPoint getSkewness() const throw(NotDefinedException);

	/** Get the kurtosis of the distribution */
	NumericalPoint getKurtosis() const throw(NotDefinedException);

	/** Get the covariance of the distribution */
	CovarianceMatrix getCovariance() const throw(NotDefinedException);

	/** Parameters value and description accessor */
	NumericalPointWithDescriptionCollection getParametersCollection() const;


	/* Interface specific to Triangular */

	/** A accessor */
	void setAMB(const NumericalScalar a,
                    const NumericalScalar m,
                    const NumericalScalar b)
	  throw (InvalidArgumentException);
	NumericalScalar getA() const;

	/** M accessor */
	NumericalScalar getM() const;

	/** B accessor */
	NumericalScalar getB() const;

	/** Method save() stores the object through the StorageManager */
	void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(const StorageManager::Advocate & adv);


      protected:
    
    
      private:

	/** Compute the numerical range of the distribution given the parameters values */
	void computeRange();

	/** The main parameter set of the distribution */
	NumericalScalar a_;
	NumericalScalar m_;
	NumericalScalar b_;

      }; /* class Triangular */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_TRIANGULAR_HXX */
