//                                               -*- C++ -*-
/**
 *  @file  NonCentralStudent.cxx
 *  @brief The NonCentralStudent distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-13 22:37:56 +0200 (sam 13 sep 2008) $
 *  Id:      $Id: NonCentralStudent.cxx 929 2008-09-13 20:37:56Z dutka $
 */
#include <cmath>
#include "NonCentralStudent.hxx"
#include "RandomGenerator.hxx"
#include "SpecFunc.hxx"
#include "DistFunc.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      CLASSNAMEINIT(NonCentralStudent);

      static Base::Common::Factory<NonCentralStudent> RegisteredFactory("NonCentralStudent");

      typedef Base::Stat::RandomGenerator RandomGenerator;

      /* Default constructor */
      NonCentralStudent::NonCentralStudent(const NumericalScalar nu,
					   const NumericalScalar delta,
					   const NumericalScalar gamma)
	throw(InvalidArgumentException)
	: NonEllipticalDistribution("NonCentralStudent"),
	  nu_(0.0),
	  delta_(delta),
	  gamma_(gamma)
      {
	setNu(nu);
	setDimension(1);
      }

      /* Comparison operator */
      Bool NonCentralStudent::operator ==(const NonCentralStudent & other) const
      {
	if (this != & other) return (nu_ == other.nu_) && (delta_ == other.delta_) && (gamma_ == other.gamma_);
	return true;
      }
  
      /* String converter */
      String NonCentralStudent::str() const {
	OSS oss;
	oss << "class=" << NonCentralStudent::GetClassName()
	    << " name=" << getName()
	    << " dimension=" << getDimension()
	    << " nu=" << nu_
	    << " delta=" << delta_
	    << " gamma=" << gamma_;
	return oss;
      }
  
      /* Virtual constructor */
      NonCentralStudent * NonCentralStudent::clone() const
      {
	return new NonCentralStudent(*this);
      }

      /* Get one realization of the distribution */
      NonCentralStudent::NumericalPoint NonCentralStudent::getRealization() const
      {
	return NumericalPoint(1, gamma_ + DistFunc::rNonCentralStudent(nu_, delta_));
      }
     
      /* Get the PDF of the distribution */
      NumericalScalar NonCentralStudent::computePDF(const NumericalPoint & point) const
      {
	pdfEpsilon_ = DistFunc::Precision;
	return DistFunc::dNonCentralStudent(nu_, delta_, point[0] - gamma_);
      }

      /* Get the CDF of the distribution */
      NumericalScalar NonCentralStudent::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
	cdfEpsilon_ = DistFunc::Precision;
	return DistFunc::pNonCentralStudent(nu_, delta_, point[0] - gamma_);
      }

      /** Get the PDFGradient of the distribution */
      NonCentralStudent::NumericalPoint NonCentralStudent::computePDFGradient(const NumericalPoint & point) const
      {
	NumericalScalar eps(pow(DistFunc::Precision, 1.0 / 3.0));
	NumericalPoint pdfGradient(3);
	pdfGradient[0] = (DistFunc::dNonCentralStudent(nu_ + eps, delta_, point[0] - gamma_) - DistFunc::dNonCentralStudent(nu_ - eps, delta_, point[0] - gamma_)) / (2.0 * eps);
	pdfGradient[1] = (DistFunc::dNonCentralStudent(nu_, delta_ + eps, point[0] - gamma_) - DistFunc::dNonCentralStudent(nu_, delta_ - eps, point[0] - gamma_)) / (2.0 * eps);
	pdfGradient[2] = (DistFunc::dNonCentralStudent(nu_, delta_, point[0] - gamma_ - eps) - DistFunc::dNonCentralStudent(nu_, delta_, point[0] - gamma_ + eps)) / (2.0 * eps);
	return pdfGradient;
      }

      /** Get the CDFGradient of the distribution */
      NonCentralStudent::NumericalPoint NonCentralStudent::computeCDFGradient(const NumericalPoint & point) const
      {
	NumericalScalar eps(pow(DistFunc::Precision, 1.0 / 3.0));
	NumericalPoint cdfGradient(3);
	cdfGradient[0] = (DistFunc::pNonCentralStudent(nu_ + eps, delta_, point[0] - gamma_) - DistFunc::pNonCentralStudent(nu_ - eps, delta_, point[0] - gamma_)) / (2.0 * eps);
	cdfGradient[1] = (DistFunc::pNonCentralStudent(nu_, delta_ + eps, point[0] - gamma_) - DistFunc::pNonCentralStudent(nu_, delta_ - eps, point[0] - gamma_)) / (2.0 * eps);
	cdfGradient[2] = (DistFunc::pNonCentralStudent(nu_, delta_, point[0] - gamma_ - eps) - DistFunc::pNonCentralStudent(nu_, delta_, point[0] - gamma_ + eps)) / (2.0 * eps);
	return cdfGradient;
      }

      /* Get the mean of the distribution */
      NonCentralStudent::NumericalPoint NonCentralStudent::getMean() const throw(NotDefinedException)
      {
	if (nu_ <= 1.0) throw NotDefinedException(HERE) << "Error: the mean is defined only for nu > 1 for a non central Student distribution";
	return NumericalPoint(1, sqrt(0.5 * nu_) * exp(SpecFunc::LnGamma(0.5 * (nu_ - 1.0)) - SpecFunc::LnGamma(0.5 * nu_)) * delta_ + gamma_);
      }

      /* Get the standard deviation of the distribution */
      NonCentralStudent::NumericalPoint NonCentralStudent::getStandardDeviation() const throw(NotDefinedException)
      {
	if (nu_ <= 2.0) throw NotDefinedException(HERE) << "Error: the standard deviation is defined only for nu > 2 for a non central Student distribution";
	return NumericalPoint(1, sqrt(getCovariance().operator()(0, 0)));
      }

      /* Get the skewness of the distribution */
      NonCentralStudent::NumericalPoint NonCentralStudent::getSkewness() const throw(NotDefinedException)
      {
	NumericalScalar mup1(getMean()[0] - gamma_);
	NumericalScalar mu2(getCovariance().operator()(0, 0));
	return NumericalPoint(1, mup1 * (nu_ * (2.0 * nu_ - 3.0 + delta_ * delta_) / ((nu_ - 2.0) * (nu_ - 3.0)) - 2.0 * mu2) * pow(mu2, -1.5));
      }

      /* Get the kurtosis of the distribution */
      NonCentralStudent::NumericalPoint NonCentralStudent::getKurtosis() const throw(NotDefinedException)
      {
	NumericalScalar mup1(getMean()[0] - gamma_);
	NumericalScalar mu2(getCovariance().operator()(0, 0));
	NumericalScalar delta2(delta_ * delta_);
	return NumericalPoint(1, (nu_ * nu_ * (3.0 + 6.0 * delta2 + delta2 * delta2) / ((nu_ - 2.0) * (nu_ - 4.0)) - mup1 * mup1 * (nu_ * ((nu_ + 1.0) * delta2 + 3.0 * (3.0 * nu_ - 5.0)) / ((nu_ - 2.0) * (nu_ - 3.0)) - 3.0 * mu2)) * pow(mu2, -2.0));
      }

      /* Get the covariance of the distribution */
      NonCentralStudent::CovarianceMatrix NonCentralStudent::getCovariance() const throw(NotDefinedException)
      {
	if (nu_ <= 2.0) throw NotDefinedException(HERE) << "Error: the covariance is defined only for nu > 2 for a non central Student distribution";
	CovarianceMatrix covariance(1);
	NumericalScalar mup1(getMean()[0] - gamma_);
	covariance(0, 0) =  nu_ / (nu_ - 2.0) * (1.0 + delta_ * delta_) - mup1 * mup1;
	return covariance;
      }

      /* Parameters value and description accessor */
      NonCentralStudent::NumericalPointWithDescriptionCollection NonCentralStudent::getParametersCollection() const
      {
	NumericalPointWithDescriptionCollection parameters(1);
	NumericalPointWithDescription point(3);
        Description description(point.getDimension());
	point[0] = nu_;
	point[1] = delta_;
	point[2] = gamma_;
	description[0] = "nu";
	description[1] = "delta";
	description[2] = "gamma";
	point.setDescription(description);
	point.setName(getDescription()[0]);
	parameters[0] = point;
	return parameters;
      }

      /* Nu accessor */
      void NonCentralStudent::setNu(const NumericalScalar nu)
	throw(InvalidArgumentException)
      {
	if (nu <= 0.0) throw InvalidArgumentException(HERE) << "Nu MUST be strictly positive";
	nu_ = nu;
      }

      /* Nu accessor */
      NumericalScalar NonCentralStudent::getNu() const
      {
	return nu_;
      }


      /* Delta accessor */
      void NonCentralStudent::setDelta(const NumericalScalar delta)
      {
	delta_ = delta;
      }

      /* Gamma accessor */
      NumericalScalar NonCentralStudent::getDelta() const
      {
	return delta_;
      }

      /* Gamma accessor */
      void NonCentralStudent::setGamma(const NumericalScalar gamma)
      {
	gamma_ = gamma;
      }

      /* Gamma accessor */
      NumericalScalar NonCentralStudent::getGamma() const
      {
	return gamma_;
      }

      /* Method save() stores the object through the StorageManager */
      void NonCentralStudent::save(const StorageManager::Advocate & adv) const
      {
	NonEllipticalDistribution::save(adv);
	adv.writeValue("nu_", nu_);
	adv.writeValue("delta_", delta_);
	adv.writeValue("gamma_", gamma_);
      }

      /* Method load() reloads the object from the StorageManager */
      void NonCentralStudent::load(const StorageManager::Advocate & adv)
      {
	NonEllipticalDistribution::load(adv);
	String name;
	NumericalScalar value;
	StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead())
	  {
	    if (objList.readValue(name, value))
	      {
		if (name == "nu_") nu_ = value;
		if (name == "delta_") delta_ = value;
		if (name == "gamma_") gamma_ = value;
	      }
	  }
      }
      
    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
