//                                               -*- C++ -*-
/**
 *  @file  KernelMixture.cxx
 *  @brief Class for a product-kernel multidimensional mixture. If K is the
 *         underlying 1D kernel, h=(h_1,\dot,h_n) is the vector of bandwidth
 *         and X=(X^1,\dots,X^N) is the nD sample, the PDF of the kernel mixture
 *         is:
 *         PDF(x) = C\sum_{i=1}^N\prod_{j=1}^n K((X^i_j-x_j)/h_j),
 *         where:
 *         C = \frac{1}{N\prod_{k=1}^n h_k}
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: KernelMixture.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include <cmath>
#include "KernelMixture.hxx"
#include "RandomGenerator.hxx"
#include "Exception.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Stat::RandomGenerator              RandomGenerator;
      typedef Base::Common::NotYetImplementedException NotYetImplementedException;

      CLASSNAMEINIT(KernelMixture);

      static Base::Common::Factory<KernelMixture> RegisteredFactory("KernelMixture");

      /* Default constructor */
      KernelMixture::KernelMixture(const Distribution & kernel,
                                   const NumericalPoint & bandwidth,
                                   const NumericalSample & sample)
	throw (InvalidArgumentException)
	: NonEllipticalDistribution("KernelMixture"),
	  kernel_(kernel),
	  bandwidth_(0),
	  bandwidthInverse_(0),
	  normalizationFactor_(0.0),
	  sample_(sample),
	  mean_(bandwidth.getDimension()),
	  isAlreadyComputedMean_(false),
	  covariance_(bandwidth.getDimension()),
	  isAlreadyComputedCovariance_(false)
      {
	// We check if the given kernel is 1-D (product kernel)
	if (kernel.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: only 1D kernel is allowed for multidimensional product kernels";
	if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a KernelMixture based on an empty sample.";
	setDimension(sample.getDimension());
	setBandwidth(bandwidth);
	computeRange();
      }

      /* Comparison operator */
      Bool KernelMixture::operator ==(const KernelMixture & other) const {
	Bool sameObject = false;

	if (this != &other) { // Other is NOT me, so I have to realize the comparison
	  // sameObject = ...
	} else sameObject = true;

	return sameObject;
      }
  
      /* String converter */
      String KernelMixture::str() const {
	OSS oss;
	oss << "class=" << KernelMixture::GetClassName()
	    << " name=" << getName()
	    << " kernel=" << kernel_
	    << " bandwidth=" << bandwidth_
	    << " sample=" << sample_;
	return oss;
      }



      /* Compute the numerical range of the distribution given the parameters values */
      void KernelMixture::computeRange()
      {
	const Interval kernelRange(kernel_.getRange());
	const UnsignedLong dimension(getDimension());
	const NumericalPoint lowerBound(sample_.getMin() + kernelRange.getLowerBound()[0] * bandwidth_);
	const NumericalPoint upperBound(sample_.getMax() + kernelRange.getUpperBound()[0] * bandwidth_);
	const Interval::BoolCollection finiteLowerBound(dimension, kernelRange.getFiniteLowerBound()[0]);
	const Interval::BoolCollection finiteUpperBound(dimension, kernelRange.getFiniteUpperBound()[0]);
	setRange(Interval(lowerBound, upperBound, finiteLowerBound, finiteUpperBound));
      }

      /* Kernel accessor */
      void KernelMixture::setKernel(const Distribution & kernel)
	throw (InvalidArgumentException)
      {
	// We check if the kernel is 1D
	if (kernel.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: the kernel must be 1D for product kernel mixture";
	kernel_ = kernel;
	isAlreadyComputedMean_ = false;
	isAlreadyComputedCovariance_ = false;
	computeRange();
      }

      KernelMixture::Distribution KernelMixture::getKernel() const
      {
	return kernel_;
      }

      /* Sample accessor */
      void KernelMixture::setSample(const NumericalSample & sample)
	throw (InvalidArgumentException)
      {
	if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a KernelMixture based on an empty sample.";
	sample_ = sample;
	isAlreadyComputedMean_ = false;
	isAlreadyComputedCovariance_ = false;
	computeRange();
      }

      KernelMixture::NumericalSample KernelMixture::getSample() const
      {
	return sample_;
      }

      
      /* Bandwidth accessor */
      void KernelMixture::setBandwidth(const NumericalPoint & bandwidth)
	throw (InvalidArgumentException)
      {
	UnsignedLong dimension(getDimension());
	normalizationFactor_ = sample_.getSize();
	if (bandwidth.getDimension() != dimension) throw InvalidArgumentException(HERE) << "Error: the dimensions of the bandwidth and the sample must be equal";
	bandwidthInverse_ = NumericalPoint(dimension);
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    NumericalScalar hi(bandwidth[i]);
            if (hi <= 0.0) throw InvalidArgumentException(HERE) << "Error: the bandwidth components must be > 0, here bandwidth=" << bandwidth;
	    bandwidthInverse_[i] = 1.0 / hi;
	    normalizationFactor_ *= hi;
	  }
	bandwidth_ = bandwidth;
	normalizationFactor_ = 1.0 / normalizationFactor_;
	isAlreadyComputedMean_ = false;
	isAlreadyComputedCovariance_ = false;
	computeRange();
      }

      /* Distribution collection accessor */
      KernelMixture::NumericalPoint KernelMixture::getBandwidth() const
      {
	return bandwidth_;
      }

      /* Virtual constructor */
      KernelMixture * KernelMixture::clone() const
      {
	return new KernelMixture(*this);
      }

      /* Get one realization of the KernelMixture */
      KernelMixture::NumericalPoint KernelMixture::getRealization() const
      {
	// Select the atom uniformly amongst the possible points
	NumericalPoint result(sample_[RandomGenerator::IntegerGenerate(sample_.getSize())]);
	// Then add a random noise according to the product kernel
	UnsignedLong dimension(getDimension());
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    result[i] -= bandwidth_[i] * kernel_.getRealization()[0];
	  }
	return result;
      }

      /* Get the DDF of the KernelMixture */
      KernelMixture::NumericalPoint KernelMixture::computeDDF(const NumericalPoint & point) const 
      {
	const UnsignedLong dimension(getDimension());
	NumericalPoint ddfValue(dimension, 0.0);
	// Quick rejection test
	if (!getRange().isInside(point)) return ddfValue;
	const UnsignedLong size(sample_.getSize());
	for(UnsignedLong i = 0; i < size; ++i)
	  {
	    NumericalPoint atom(dimension, 0.0);
	    NumericalPoint kernelPdfAtom(dimension, 0.0);
	    NumericalScalar pdfAtom(1.0);
	    for (UnsignedLong j = 0; j < dimension; ++j)
	      {
		atom[j] = (sample_[i][j] - point[j]) * bandwidthInverse_[j];
		kernelPdfAtom[j] = kernel_.computePDF(NumericalPoint(1, atom[j]));
		pdfAtom *= kernelPdfAtom[j];
	      }
	    for (UnsignedLong j = 0; j < dimension; ++j)
	      {
		ddfValue[j] -= pdfAtom / kernelPdfAtom[j] * kernel_.computeDDF(NumericalPoint(1, atom[j]))[0] * bandwidthInverse_[j];
	      }
	  } /* end for */
	return normalizationFactor_ * ddfValue;
      }

      /* Get the PDF of the KernelMixture */
      NumericalScalar KernelMixture::computePDF(const NumericalPoint & point) const 
      {
	const UnsignedLong dimension(getDimension());
	NumericalScalar pdfValue(0.0);
	// Quick rejection test
	if (!getRange().isInside(point)) return pdfValue;
	const UnsignedLong size(sample_.getSize());
	for(UnsignedLong i = 0; i < size; ++i)
	  {
	    NumericalScalar pdfAtom(1.0);
	    for (UnsignedLong j = 0; j < dimension; ++j)
	      {
		pdfAtom *= kernel_.computePDF(NumericalPoint(1, (sample_[i][j] - point[j]) * bandwidthInverse_[j]));
	      }
	    pdfValue += pdfAtom;
	  } /* end for */
	return normalizationFactor_ * pdfValue;
      }

      /* Get the CDF of the KernelMixture */
      NumericalScalar KernelMixture::computeCDF(const NumericalPoint & point, const Bool tail) const 
      {
	const UnsignedLong dimension(getDimension());
	NumericalScalar cdfValue(0.0);
	const UnsignedLong size(sample_.getSize());
	for(UnsignedLong i = 0; i < size; ++i)
	  {
	    NumericalScalar cdfAtom(1.0);
	    for (UnsignedLong j = 0; j < dimension; ++j)
	      {
		cdfAtom *= (1.0 - kernel_.computeCDF(NumericalPoint(1, (sample_[i][j] - point[j]) * bandwidthInverse_[j])));
	      }
	    cdfValue += cdfAtom;
	  } /* end for */
	cdfEpsilon_ = kernel_.getCDFEpsilon() * size;
	return cdfValue / sample_.getSize();
      }

      /* Get the PDF gradient of the distribution */
      KernelMixture::NumericalPoint KernelMixture::computePDFGradient(const NumericalPoint & point) const
      {
	throw NotYetImplementedException(HERE);
      }

      /* Get the CDF gradient of the distribution */
      KernelMixture::NumericalPoint KernelMixture::computeCDFGradient(const NumericalPoint & point) const
      {
	throw NotYetImplementedException(HERE);
      }

      /* Get the i-th marginal distribution */
      KernelMixture::Implementation KernelMixture::getMarginal(const UnsignedLong i) const throw(InvalidArgumentException)
      {
	UnsignedLong dimension(getDimension());
	if (i >= dimension) throw InvalidArgumentException(HERE) << "The index of a marginal distribution must be in the range [0, dim-1]";
	// Special case for dimension 1
	if (dimension == 1) return clone();
	// General case
	KernelMixture kernelMixture(kernel_, NumericalPoint(1, bandwidth_[i]), sample_.getMarginal(i));
	kernelMixture.setName("Marginal of " + getName());
	return kernelMixture.clone();
      }

      /* Get the distribution of the marginal distribution corresponding to indices dimensions */
      KernelMixture::Implementation KernelMixture::getMarginal(const Indices & indices) const throw(InvalidArgumentException)
      {
	UnsignedLong dimension(getDimension());
	if (!indices.check(dimension - 1)) throw InvalidArgumentException(HERE) << "The indices of a marginal distribution must be in the range [0, dim-1] and  must be different";
	// Special case for dimension 1
	if (dimension == 1) return clone();
	// General case
	UnsignedLong marginalDimension(indices.getSize());
	NumericalPoint marginalBandwidth(marginalDimension);
	for (UnsignedLong i = 0; i < marginalDimension; ++i)
	  {
	    marginalBandwidth[i] = bandwidth_[indices[i]];
	  }
	KernelMixture kernelMixture(kernel_, marginalBandwidth, sample_.getMarginal(indices));
	kernelMixture.setName("Marginal of " + getName());
	return kernelMixture.clone();
      }

      /* Compute the mean of the KernelMixture
	 mean(KernelMixture) = mean(sample) - mean(kernel) * bandwidth
      */
      void KernelMixture::computeMean() const throw(NotDefinedException)
      {
	// We know that the kernel is 1D, so its mean value is actually a scalar
	NumericalScalar meanKernel(kernel_.getMean()[0]);
	mean_ = sample_.computeMean();
	// Special case for symmetric kernel
	if (meanKernel == 0.0) return;
	// General case
        mean_ -= meanKernel * bandwidthInverse_;
	isAlreadyComputedMean_ = true;
      }

      /* Get the mean of the distribution */
      KernelMixture::NumericalPoint KernelMixture::getMean() const throw(NotDefinedException)
      {
	if (!isAlreadyComputedMean_)
	  {
	    computeMean();
	  }
	return mean_;
      }

      /* Compute the covariance of the KernelMixture
	 Covariance(KernelMixture) = (1-1/N) Covariance(sample) + mean(kernel)^2*bandwidth*transpose(bandwidth) - mean(K) * (bandwidth * transpose(mean(sample)) + mean(sample) * transpose(mean(kernel))) + Covariance(kernel) * diag(bandwidth[i]^2)
      */
      void KernelMixture::computeCovariance() const throw(NotDefinedException)
      {
	UnsignedLong dimension(getDimension());
	// We know that the kernel is 1D, so its covariance is actually a scalar
	NumericalScalar covarianceKernel(kernel_.getCovariance()(0, 0));
	// Covariance(sample) term
	covariance_ = sample_.computeCovariance();
	// We know that the kernel is 1D, so its mean is actually a scalar
	NumericalScalar meanKernel(kernel_.getMean()[0]);
	NumericalPoint mean(getMean());
	NumericalScalar scaling(1.0 - 1.0 / sample_.getSize());
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    NumericalScalar h_i(bandwidth_[i]);
	    NumericalScalar mu_i(mean[i]);
	    // Upper-triangular terms, including the diagonal
	    for (UnsignedLong j = i; j < dimension; ++j)
	      {
		NumericalScalar h_j(bandwidth_[j]);
		NumericalScalar mu_j(mean[j]);
		// We factor the mean(kernel) term
		covariance_(i, j) = scaling * covariance_(i, j) + meanKernel * (meanKernel * h_i * h_j - h_i * mu_j - h_j * mu_i);
	      }
	    covariance_(i, i) += covarianceKernel * h_i * h_i;
	  }
	isAlreadyComputedCovariance_ = true;
      }

      /* Get the covariance of the distribution */
      KernelMixture::CovarianceMatrix KernelMixture::getCovariance() const throw(NotDefinedException)
      {
	if (!isAlreadyComputedCovariance_)
	  {
	    computeCovariance();
	  }
	return covariance_;
      }

      /** Parameters value and description accessor */
      KernelMixture::NumericalPointWithDescriptionCollection KernelMixture::getParametersCollection() const
      {
	const UnsignedLong dimension(getDimension());
	const UnsignedLong size(sample_.getSize());
	NumericalPointWithDescriptionCollection parameters(dimension + 1);
	// The marginal parameters : the sample and the bandwidth
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    NumericalPointWithDescription marginalParameters(size + 1);
	    for (UnsignedLong j = 0; j < size; ++j)
	      {
		marginalParameters[j] = sample_[j][i];
	      }
	    marginalParameters[size] = bandwidth_[i];
	    parameters[i] = marginalParameters;
	  }
	parameters[dimension] = kernel_.getParametersCollection()[0];
	return parameters;
      } // getParametersCollection

      /* Check if the distribution is elliptical */
      Bool KernelMixture::isElliptical() const
      {
	if ((sample_.getSize() == 1) && (getDimension() == 1)) return kernel_.getImplementation()->isElliptical();
	return false;
      }

      /* Check if the distribution is continuos */
      Bool KernelMixture::isContinuous() const
      {
	return kernel_.getImplementation()->isContinuous();
      }

      /* Tell if the distribution has elliptical copula */
      Bool KernelMixture::hasEllipticalCopula() const
      {
	// In 1D, all the distributions have an elliptical copula
	if (getDimension() == 1) return true;
	return false;
      }

      /* Tell if the distribution has independent copula */
      Bool KernelMixture::hasIndependentCopula() const
      {
	// In 1D, all the distributions have an independent copula
	if (getDimension() == 1) return true;
	return false;
      }

      /* Method save() stores the object through the StorageManager */
      void KernelMixture::save(const StorageManager::Advocate & adv) const
      {
	NonEllipticalDistribution::save(adv);
	adv.writeValue(kernel_, StorageManager::MemberNameAttribute, "kernel_");
	adv.writeValue(bandwidth_, StorageManager::MemberNameAttribute, "bandwidth_");
	adv.writeValue(bandwidthInverse_, StorageManager::MemberNameAttribute, "bandwidthInverse_");
	adv.writeValue("normalizationFactor_", normalizationFactor_);
	adv.writeValue(sample_, StorageManager::MemberNameAttribute, "sample_");
	adv.writeValue(mean_, StorageManager::MemberNameAttribute, "mean_");
	adv.writeValue("isAlreadyComputedMean_", isAlreadyComputedMean_);
	adv.writeValue(covariance_, StorageManager::MemberNameAttribute, "covariance_");
	adv.writeValue("isAlreadyComputedCovariance_", isAlreadyComputedCovariance_);
      }

      /* Method load() reloads the object from the StorageManager */
      void KernelMixture::load(const StorageManager::Advocate & adv)
      {
	NonEllipticalDistribution::load(adv);

	// Read the atomic NumericalScalar attributes
	{
	  String name;
	  NumericalScalar value;
	  StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
	  for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	    if (objList.readValue(name, value)) {
	      if (name == "normalizationFactor_") normalizationFactor_ = value;
	    }
	  }
	}
	// Read the atomic Bool attributes
	{
	  String name;
	  Bool value;
	  StorageManager::List objList = adv.getList(StorageManager::BoolEntity);
	  for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	    if (objList.readValue(name, value)) {
	      if (name == "isAlreadyComputedMean_") isAlreadyComputedMean_ = value;
	      if (name == "isAlreadyComputedCovariance_") isAlreadyComputedCovariance_ = value;
	    }
	  }
	}
	// Read the objects attributes
	adv.readValue(kernel_, StorageManager::MemberNameAttribute, "kernel_");
	adv.readValue(bandwidth_, StorageManager::MemberNameAttribute, "bandwidth_");
	adv.readValue(bandwidthInverse_, StorageManager::MemberNameAttribute, "bandwidthInverse_");
	adv.readValue(sample_, StorageManager::MemberNameAttribute, "sample_");
	adv.readValue(mean_, StorageManager::MemberNameAttribute, "mean_");
	adv.readValue(covariance_, StorageManager::MemberNameAttribute, "covariance_");
	computeRange();
      }
      
    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
