//                                               -*- C++ -*-
/**
 *  @file  IndependentCopula.cxx
 *  @brief A class that implements an independent copula
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: IndependentCopula.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include <cmath>
#include "IndependentCopula.hxx"
#include "Normal.hxx"
#include "IdentityMatrix.hxx"
#include "NatafIndependentCopulaEvaluation.hxx"
#include "NatafIndependentCopulaGradient.hxx"
#include "NatafIndependentCopulaHessian.hxx"
#include "InverseNatafIndependentCopulaEvaluation.hxx"
#include "InverseNatafIndependentCopulaGradient.hxx"
#include "InverseNatafIndependentCopulaHessian.hxx"
#include "PersistentObjectFactory.hxx"
#include "RandomGenerator.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Type::IdentityMatrix                         IdentityMatrix;
      typedef Base::Stat::NumericalSample                        NumericalSample;
      typedef Base::Stat::RandomGenerator                        RandomGenerator;
      typedef Algorithm::NatafIndependentCopulaEvaluation        NatafIndependentCopulaEvaluation;
      typedef Algorithm::NatafIndependentCopulaGradient          NatafIndependentCopulaGradient;
      typedef Algorithm::NatafIndependentCopulaHessian           NatafIndependentCopulaHessian;
      typedef Algorithm::InverseNatafIndependentCopulaEvaluation InverseNatafIndependentCopulaEvaluation;
      typedef Algorithm::InverseNatafIndependentCopulaGradient   InverseNatafIndependentCopulaGradient;
      typedef Algorithm::InverseNatafIndependentCopulaHessian    InverseNatafIndependentCopulaHessian;

      CLASSNAMEINIT(IndependentCopula);

      static Base::Common::Factory<IndependentCopula> RegisteredFactory("IndependentCopula");

      /* Default constructor */
      IndependentCopula::IndependentCopula(const UnsignedLong dimension)
	: CopulaImplementation("IndependentCopula")
      {
	// Initialize any other class members here
	// At last, allocate memory space if needed, but go to destructor to free it
	setDimension(dimension);
      }

      /* Comparison operator */
      Bool IndependentCopula::operator ==(const IndependentCopula & other) const
      {
	Bool sameObject = false;

	if (this != &other) { // Other is NOT me, so I have to realize the comparison
	  // sameObject = ...
	  // TODO : Write IndependentCopula::operator ==()
	  if (CopulaImplementation::operator==(other)) {
	    sameObject = true;
	  }

	} else sameObject = true;

	return sameObject;
      }
  
      /* String converter */
      String IndependentCopula::str() const {
	OSS oss;
	oss << "class=" << IndependentCopula::GetClassName()
	    << " name=" << getName()
	    << " dimension=" << getDimension();
	return oss;
      }
  
      /* Virtual constructor */
      IndependentCopula * IndependentCopula::clone() const
      {
	return new IndependentCopula(*this);
      }

      /* Get one realization of the distribution */
      IndependentCopula::NumericalPoint IndependentCopula::getRealization() const
      {
	return RandomGenerator::Generate(getDimension());
      }

      /* Get the DDF of the distribution */
      IndependentCopula::NumericalPoint IndependentCopula::computeDDF(const NumericalPoint & point) const
      {
	return NumericalPoint(getDimension(), 0.0);
      }

      /* Compute the probability content of an interval */
      NumericalScalar IndependentCopula::computeProbability(const Interval & interval) const
      {
	const UnsignedLong dimension(getDimension());
	// Reduce the given interval to the support of the distribution, which is the nD unit cube
	const Interval intersect(interval.intersect(Interval(dimension)));
	// If the intersection is empty
	if (intersect.isEmpty()) return 0.0;
	const NumericalPoint lower(intersect.getLowerBound());
	const NumericalPoint upper(intersect.getUpperBound());
	NumericalScalar value(1.0);
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    value *= upper[i] - lower[i];
	  }
	return value;
      }

      /* Get the PDF of the distribution */
      NumericalScalar IndependentCopula::computePDF(const NumericalPoint & point) const
      {
	UnsignedLong dimension(getDimension());
	for (UnsignedLong i = 0; i < dimension; i++)
	  {
	    NumericalScalar x(point[i]);
	    // If one component is outside of the support, the PDF is null
	    if ((x <= 0.0) || (x >= 1.0)) return 0.0;
	  }
	// The point is in the support
	return 1.0;
      }

      /* Get the CDF of the distribution */
      NumericalScalar IndependentCopula::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
	UnsignedLong dimension(getDimension());
	NumericalScalar value(1.0);
	for (UnsignedLong i = 0; i < dimension; i++)
	  {
	    NumericalScalar x(point[i]);
	    // If one component is at the left of the support of its matginal distribution, the CDF is null
	    if (x <= 0.0) return 0.0;
	    // If the component is inside of the support, multiply the value of the CDF by x
	    // FIXME
	    if (x < 1.0) value *= x;
	  }
	return value;
      }

      /* Get the PDF gradient of the distribution */
      IndependentCopula::NumericalPoint IndependentCopula::computePDFGradient(const NumericalPoint & point) const
      {
	return NumericalPoint(0, 0.0);
      }

      /* Get the CDF gradient of the distribution */
      IndependentCopula::NumericalPoint IndependentCopula::computeCDFGradient(const NumericalPoint & point) const
      {
	return NumericalPoint(0, 0.0);
      }


      /* Get the quantile of the distribution */
      IndependentCopula::NumericalPoint IndependentCopula::computeQuantile(const NumericalScalar prob) const 
      {
	if (prob < 0.0 || prob > 1.0) throw InvalidArgumentException(HERE) << "Error: cannot compute a quantile for a probability level outside of [0, 1]";
	if (prob == 0.0) return getRange().getLowerBound();
	if (prob == 1.0) return getRange().getUpperBound();
	return NumericalPoint(getDimension(), pow(prob, 1.0 / getDimension()));
      }

      /* Get the distribution of the marginal distribution corresponding to indices dimensions */
      IndependentCopula::Implementation IndependentCopula::getMarginal(const Indices & indices) const throw(InvalidArgumentException)
      {
      	UnsignedLong dimension(getDimension());
	if (!indices.check(dimension - 1)) throw InvalidArgumentException(HERE) << "Error: the indices of a marginal distribution must be in the range [0, dim-1] and  must be different";
	// General case
      	UnsignedLong outputDimension(indices.getSize());
	// Special case if indices is of length 1
	if (outputDimension ==1) return getMarginal(indices[0]);
	IndependentCopula result(outputDimension);
	result.setName("Marginal of " + getName());
	result.setWeight(getWeight());
      	return new IndependentCopula(result);
      }

      /* Get the standard distribution, i.e. a distribution of the same kind but with zero mean,
       * unit marginal standard distribution and identity correlation */
      IndependentCopula::Implementation IndependentCopula::getStandardDistribution() const
      {
	Normal standard(getDimension());
	standard.setDescription(getDescription());
	return new Normal(standard);
      }

      /* Get the covariance of the distribution */
      IndependentCopula::CovarianceMatrix IndependentCopula::getCovariance() const throw(OT::Base::Common::NotDefinedException)
      {
	UnsignedLong dimension(getDimension());
	CovarianceMatrix covariance(dimension);
	NumericalScalar twelveInverse(1.0 / 12.0);
	for (UnsignedLong i = 0; i < dimension; i++)
	  {
	    covariance(i, i) = twelveInverse;
	  }
	return covariance;
      }

      /* Compute the DDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
      NumericalScalar IndependentCopula::computeConditionalDDF(const NumericalScalar x, const NumericalPoint & y) const
      {
	return 0.0;
      }
      
      /* Compute the PDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
      NumericalScalar IndependentCopula::computeConditionalPDF(const NumericalScalar x, const NumericalPoint & y) const
      {
	const UnsignedLong conditioningDimension(y.getDimension());
	if (conditioningDimension >= getDimension()) throw InvalidArgumentException(HERE) << "Error: cannot compute a conditional PDF with a conditioning point of dimension greater or equal to the distribution dimension.";
	if (x < 0.0) return 0.0;
	if (x < 1.0) return 1.0;
	return 0.0;
      }
      
      /* Compute the CDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
      NumericalScalar IndependentCopula::computeConditionalCDF(const NumericalScalar x, const NumericalPoint & y) const
      {
	const UnsignedLong conditioningDimension(y.getDimension());
	if (conditioningDimension >= getDimension()) throw InvalidArgumentException(HERE) << "Error: cannot compute a conditional quantile with a conditioning point of dimension greater or equal to the distribution dimension.";
	if (x < 0.0) return 0.0;
	if (x < 1.0) return x;
	return 1.0;
      }
      
      /* Compute the quantile of Xi | X1, ..., Xi-1, i.e. x such that CDF(x|y) = q with x = Xi, y = (X1,...,Xi-1) */
      NumericalScalar IndependentCopula::computeConditionalQuantile(const NumericalScalar q, const NumericalPoint & y) const
      {
	const UnsignedLong conditioningDimension(y.getDimension());
	if (conditioningDimension >= getDimension()) throw InvalidArgumentException(HERE) << "Error: cannot compute a conditional quantile with a conditioning point of dimension greater or equal to the distribution dimension.";
	if ((q < 0.0) || (q > 1.0)) throw InvalidArgumentException(HERE) << "Error: cannot compute a conditional quantile for a probability level outside of [0, 1]";
	return q;
      }

      /* Get the isoprobabilist transformation */
      IndependentCopula::IsoProbabilisticTransformation IndependentCopula::getIsoProbabilisticTransformation() const
      {
	IsoProbabilisticTransformation transformation;
	transformation.setEvaluationImplementation(new NatafIndependentCopulaEvaluation(getDimension()));
	transformation.setGradientImplementation(new NatafIndependentCopulaGradient(getDimension()));
	transformation.setHessianImplementation(new NatafIndependentCopulaHessian(getDimension()));

	return transformation;
      }

      /* Get the inverse isoprobabilistic transformation */
      IndependentCopula::InverseIsoProbabilisticTransformation IndependentCopula::getInverseIsoProbabilisticTransformation() const
      {
	InverseIsoProbabilisticTransformation transformation;
	transformation.setEvaluationImplementation(new InverseNatafIndependentCopulaEvaluation(getDimension()));
	transformation.setGradientImplementation(new InverseNatafIndependentCopulaGradient(getDimension()));
	transformation.setHessianImplementation(new InverseNatafIndependentCopulaHessian(getDimension()));

	return transformation;
      }

      /** Parameters value and description accessor */
      IndependentCopula::NumericalPointWithDescriptionCollection IndependentCopula::getParametersCollection() const
      {
	// No parameter
	return NumericalPointWithDescriptionCollection(0);
      }

      /* Method save() stores the object through the StorageManager */
      void IndependentCopula::save(const StorageManager::Advocate & adv) const
      {
	CopulaImplementation::save(adv);
      }

      /* Method load() reloads the object from the StorageManager */
      void IndependentCopula::load(const StorageManager::Advocate & adv)
      {
	CopulaImplementation::load(adv);
      }
      
    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
