//                                               -*- C++ -*-
/**
 *  @file  Gamma.cxx
 *  @brief The Gamma distribution
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-13 22:37:56 +0200 (sam 13 sep 2008) $
 *  Id:      $Id: Gamma.cxx 929 2008-09-13 20:37:56Z dutka $
 */
#include <cmath>
#include "Gamma.hxx"
#include "RandomGenerator.hxx"
#include "SpecFunc.hxx"
#include "DistFunc.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Stat::RandomGenerator RandomGenerator;

      CLASSNAMEINIT(Gamma);

      static Base::Common::Factory<Gamma> RegisteredFactory("Gamma");

      /* Default constructor */
      Gamma::Gamma()
	: NonEllipticalDistribution("Gamma"),
	  k_(1.0), lambda_(1.0), gamma_(0.0), normalizationFactor_(1.0)
      {
	setDimension(1);
	computeRange();
      }

      /* Parameters constructor */
      Gamma::Gamma(const NumericalScalar arg1,
		   const NumericalScalar arg2,
		   const NumericalScalar gamma,
		   const ParameterSet set)
	throw (InvalidArgumentException)
	: NonEllipticalDistribution("Gamma"),
	  k_(0.), lambda_(0.), gamma_(gamma), normalizationFactor_(0.0)
      {
	switch (set) {
	case KLAMBDA:
	  setK(arg1);
	  setLambda(arg2);
	  break;
	  
	case MUSIGMA:
	  setMuSigma(arg1, arg2);
	  break;

	default:
	  throw InvalidArgumentException(HERE) << "Invalid parameter set argument";

	} /* end switch */
	setDimension(1);
	computeRange();
	update();
      }

      /* Comparison operator */
      Bool Gamma::operator ==(const Gamma & other) const
      {
	Bool sameObject = false;

	if (this != &other) { // Other is NOT me, so I have to realize the comparison
	  // sameObject = ...
	  // TODO: Write Gamma::operator ==(...)
	  if ( (k_ == other.k_) && (lambda_ == other.lambda_) && (gamma_ == other.gamma_) )
	    sameObject = true;
	} else sameObject = true;

	return sameObject;
      }
  
      /* String converter */
      String Gamma::str() const
      {
	OSS oss;
	oss << "class=" << Gamma::GetClassName()
	    << " name=" << getName()
	    << " dimension=" << getDimension()
	    << " k=" << k_
	    << " lambda=" << lambda_
	    << " gamma=" << gamma_;
	return oss;
      }
  
      /* K accessor */
      void Gamma::setK(const NumericalScalar k)
	throw(InvalidArgumentException)
      {
	if (k <= 0.) throw InvalidArgumentException(HERE) << "K MUST be positive";
	k_ = k;
	update();
      }

      NumericalScalar Gamma::getK() const
      {
	return k_;
      }


      /* E accessor */
      void Gamma::setLambda(const NumericalScalar lambda)
	throw(InvalidArgumentException)
      {
	if (lambda <= 0.) throw InvalidArgumentException(HERE) << "Lambda MUST be positive";
	lambda_ = lambda;
	computeRange();
	update();
      }

      NumericalScalar Gamma::getLambda() const
      {
	return lambda_;
      }


      /* Mu accessor */
      void Gamma::setMuSigma(const NumericalScalar mu, const NumericalScalar sigma)
      {
	lambda_ = (mu - gamma_) / sigma;
	k_ = lambda_ * lambda_;
	lambda_ /= sigma;
	computeRange();
	update();
      }
      
      NumericalScalar Gamma::getMu() const
      {
	return gamma_ + k_ / lambda_;
      }

      NumericalScalar Gamma::getSigma() const
      {
	return sqrt(k_) / lambda_;
      }


      /* Gamma accessor */
      void Gamma::setGamma(const NumericalScalar gamma)
      {
	gamma_ = gamma;
	computeRange();
      }

      NumericalScalar Gamma::getGamma() const
      {
	return gamma_;
      }





      /* Virtual constructor */
      Gamma * Gamma::clone() const
      {
	return new Gamma(*this);
      }

      /* Compute the numerical range of the distribution given the parameters values */
      void Gamma::computeRange()
      {
	// We use the range computation of the Exponential distribution because we assume that the exponential part
	// of the Gamma PDF decreases so fast that the power term has no influence
        const NumericalPoint lowerBound(1, gamma_);
        const NumericalPoint upperBound(1, gamma_ + (log(lambda_) + SpecFunc::LogMinNumericalScalar) / lambda_);
        const Interval::BoolCollection finiteLowerBound(1, true);
        const Interval::BoolCollection finiteUpperBound(1, false);
        setRange(Interval(lowerBound, upperBound, finiteLowerBound, finiteUpperBound));
      }

      /** Update the derivative attributes */
      void Gamma::update()
      {
	normalizationFactor_ = pow(lambda_, k_) / SpecFunc::Gamma(k_);
      }


      /* Get one realization of the distribution */
      Gamma::NumericalPoint Gamma::getRealization() const
      {
	return NumericalPoint(1, gamma_ + DistFunc::rGamma(k_) / lambda_);
      }
     

      /* Get the DDF of the distribution */
      Gamma::NumericalPoint Gamma::computeDDF(const NumericalPoint & point) const
      {
	NumericalScalar x(point[0] - gamma_);
	if (x <= 0.0) return NumericalPoint(1, 0.0);
	return NumericalPoint(1, ((k_ - 1.0) / x - lambda_) * computePDF(point));
      }


      /* Get the PDF of the distribution */
      NumericalScalar Gamma::computePDF(const NumericalPoint & point) const
      {
	NumericalScalar x(point[0] - gamma_);
	if (x <= 0.0) return 0.0;
	return normalizationFactor_ * pow(x, k_ - 1) * exp(-lambda_ * x);
      }


      /* Get the CDF of the distribution */
      NumericalScalar Gamma::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
	NumericalScalar x(point[0] - gamma_);
	// No test here as the CDF is continuous for all k_
	if (x <= 0.0) return (tail ? 1.0 : 0.0);
	return DistFunc::pGamma(k_, lambda_ * x, tail);
      }

      /* Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
      NumericalComplex Gamma::computeCharacteristicFunction(const NumericalPoint & point) const
      {
	NumericalScalar u(point[0]);
	return exp(NumericalComplex(0.0, -u * gamma_)) * pow(lambda_ / NumericalComplex(lambda_, -u), k_);
      }

      /* Get the PDFGradient of the distribution */
      Gamma::NumericalPoint Gamma::computePDFGradient(const NumericalPoint & point) const
      {
	NumericalPoint pdfGradient(3, 0.0);
	NumericalScalar x(point[0] - gamma_);
	if (x <= 0.0) return pdfGradient;
	NumericalScalar pdf(computePDF(point));
	pdfGradient[0] = (log(x) + log(lambda_) - SpecFunc::Psi(k_)) * pdf;
	pdfGradient[1] = (k_ / lambda_ - x) * pdf;
	pdfGradient[2] = ((1.0 - k_) / x + lambda_) * pdf;
	return pdfGradient;
      }

      /* Get the CDFGradient of the distribution */
      Gamma::NumericalPoint Gamma::computeCDFGradient(const NumericalPoint & point) const
      {
	NumericalPoint cdfGradient(3, 0.0);
	NumericalScalar x(point[0] - gamma_);
	if (x <= 0.0) return cdfGradient;
	NumericalScalar lambdaX(lambda_ * x);
	NumericalScalar factor(exp(k_ * log(lambdaX) - SpecFunc::LnGamma(k_) - lambdaX));
	NumericalScalar eps(pow(DistFunc::Precision, 1.0 / 3.0));
	cdfGradient[0] = (DistFunc::pGamma(k_ + eps, lambda_ * x) - DistFunc::pGamma(k_ - eps, lambda_ * x)) / (2.0 * eps);
	cdfGradient[1] = factor / lambda_;
	cdfGradient[2] = -factor / x;
	return cdfGradient;
      }

      /* Get the quantile of the distribution */
      Gamma::NumericalPoint Gamma::computeQuantile(const NumericalScalar prob) const
      {
	if (prob < 0.0 || prob > 1.0) throw InvalidArgumentException(HERE) << "Error: cannot compute a quantile for a probability level outside of [0, 1]";
	// Special case for boarding values
	if (prob == 0.0) return getRange().getLowerBound();
	if (prob == 1.0) return getRange().getUpperBound();
	return NumericalPoint(1, gamma_ + DistFunc::qGamma(k_, prob) / lambda_);
      }

      /* Get the mean of the distribution */
      Gamma::NumericalPoint Gamma::getMean() const throw(NotDefinedException)
      {
	return NumericalPoint(1, getMu());
      }

      /* Get the standard deviation of the distribution */
      Gamma::NumericalPoint Gamma::getStandardDeviation() const throw(NotDefinedException)
      {
	return NumericalPoint(1, getSigma());
      }

      /* Get the skewness of the distribution */
      Gamma::NumericalPoint Gamma::getSkewness() const throw(NotDefinedException)
      {
	return NumericalPoint(1, 2.0 / sqrt(k_));
      }

      /* Get the kurtosis of the distribution */
      Gamma::NumericalPoint Gamma::getKurtosis() const throw(NotDefinedException)
      {
	return NumericalPoint(1, 3.0 * (k_ + 2.0) / k_);
      }

      /* Get the covariance of the distribution */
      Gamma::CovarianceMatrix Gamma::getCovariance() const throw(NotDefinedException)
      {
	CovarianceMatrix covariance(1);
	covariance(0, 0) = k_ / (lambda_ * lambda_);
	return covariance;
      }

      /* Parameters value and description accessor */
      Gamma::NumericalPointWithDescriptionCollection Gamma::getParametersCollection() const
      {
	NumericalPointWithDescriptionCollection parameters(1);
	NumericalPointWithDescription point(3);
        Description description(point.getDimension());
	point[0] = k_;
	point[1] = lambda_;
	point[2] = gamma_;
	description[0] = "k";
	description[1] = "lambda";
	description[2] = "gamma";
	point.setDescription(description);
	point.setName(getDescription()[0]);
	parameters[0] = point;
	return parameters;
      }

      /* Method save() stores the object through the StorageManager */
      void Gamma::save(const StorageManager::Advocate & adv) const
      {
	NonEllipticalDistribution::save(adv);
	adv.writeValue("k_", k_);
	adv.writeValue("lambda_", lambda_);
	adv.writeValue("gamma_", gamma_);
	adv.writeValue("normalizationFactor_", normalizationFactor_);
      }

      /* Method load() reloads the object from the StorageManager */
      void Gamma::load(const StorageManager::Advocate & adv)
      {
	NonEllipticalDistribution::load(adv);

	String name;
	NumericalScalar value;
	StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, value)) {
	    if (name == "k_") k_ = value;
	    if (name == "lambda_") lambda_ = value;
	    if (name == "gamma_") gamma_ = value;
	    if (name == "normalizationFactor_") normalizationFactor_ = value;
	  }
	}
	computeRange();
      }
      



    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
