//                                               -*- C++ -*-
/**
 *  @file  SpecFunc.cxx
 *  @brief OpenTURNS wrapper to a library of special functions
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-13 22:37:56 +0200 (sam 13 sep 2008) $
 *  Id:      $Id: SpecFunc.cxx 929 2008-09-13 20:37:56Z dutka $
 */
#include "SpecFunc.hxx"
#include <cmath>
#include <limits>
//#undef USE_DCDFLIB
#define USE_DCDFLIB
#ifdef USE_DCDFLIB
#include "Dcdflib.hxx"
#endif
#include "Faddeeva.hxx"
#include "Debye.hxx"
#include "Exception.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      const NumericalScalar SpecFunc::Precision;

      // 0.39894228040143267 = 1 / sqrt(2.pi)
      const NumericalScalar SpecFunc::ISQRT2PI              = 0.39894228040143267;
      // 0.57721566490153286 = Euler constant gamma
      const NumericalScalar SpecFunc::EulerConstant         = 0.57721566490153286;
      // 1.64493406684822643 = pi^2 / 6
      const NumericalScalar SpecFunc::PI2_6                 = 1.64493406684822643;
      // 1.28254983016118640 = pi / sqrt(6)
      const NumericalScalar SpecFunc::PI_SQRT6              = 1.28254983016118640;
      // 0.45005320754569466 = gamma * sqrt(6) / pi
      const NumericalScalar SpecFunc::EULERSQRT6_PI         = 0.45005320754569466;
      // 3.28986813369645287 = pi^2 / 3
      const NumericalScalar SpecFunc::PI2_3                 = 3.28986813369645287;
      // 0.55132889542179204 = sqrt(3) / pi
      const NumericalScalar SpecFunc::SQRT3_PI              = 0.55132889542179204;
      // 1.81379936423421785 = pi / sqrt(3)
      const NumericalScalar SpecFunc::PI_SQRT3              = 1.81379936423421785;
      // 1.81379936423421785 = pi / sqrt(3)
      const NumericalScalar SpecFunc::ZETA3                 = 1.20205690315959429;
      // NumericalScalar limits
      const NumericalScalar SpecFunc::MinNumericalScalar    = std::numeric_limits<NumericalScalar>::min();
      const NumericalScalar SpecFunc::LogMinNumericalScalar = log(MinNumericalScalar);
      const NumericalScalar SpecFunc::MaxNumericalScalar    = std::numeric_limits<NumericalScalar>::max();
      const NumericalScalar SpecFunc::LogMaxNumericalScalar = log(MaxNumericalScalar);


      typedef Base::Common::NotYetImplementedException NotYetImplementedException;
      typedef Base::Common::InvalidArgumentException   InvalidArgumentException;

      // Beta function: Beta(a, b) = int_0^1 t^(a-1).(1-t)^(b-1) dt
      NumericalScalar SpecFunc::Beta(const NumericalScalar a, const NumericalScalar b)
      {
	return exp(lgamma(a) + lgamma(b) - lgamma(a + b));
      }

      // Incomplete Beta function: BetaInc(a, b, x) = int_0^x t^(a-1).(1-t)^(b-1) dt
      NumericalScalar SpecFunc::BetaInc(const NumericalScalar a, const NumericalScalar b, const NumericalScalar x)
      {
	return BetaRatioInc(a, b, x) * Beta(a, b);
      }

      // Incomplete Beta function inverse
      NumericalScalar SpecFunc::BetaIncInv(const NumericalScalar a, const NumericalScalar b, const NumericalScalar x)
      {
	return BetaRatioIncInv(a, b, x / Beta(a, b));
      }

      // Incomplete Beta Ratio function: BetaRatioInc(a, b, x) = 1/beta(a, b) * int_0^x t^(a-1).(1-t)^(b-1) dt
      NumericalScalar SpecFunc::BetaRatioInc(const NumericalScalar a, const NumericalScalar b, const NumericalScalar x)
      {
	if (x <= 0.0) return 0.0;
	if (x >= 1.0) return 1.0;
#ifdef USE_DCDFLIB
	double inf(a);
	double sup(b);
	double t(x);
	double y(1.0 - x);
	double w;
        double w1;
	int ierr;
	DCDFLIB::beta_inc(&inf, &sup, &t, &y, &w, &w1, &ierr);
	return w;
#else
	throw NotYetImplementedException(HERE);
#endif
      }

      // Incomplete Beta Ratio Function inverse
      NumericalScalar SpecFunc::BetaRatioIncInv(const NumericalScalar a, const NumericalScalar b, const NumericalScalar x)
      {
#ifdef USE_DCDFLIB
	int which(2);
	double inf(a);
	double sup(b);
	double p(x);
	double q(1.0 - x);
	double X;
	double Y;
	int status;
	double bound;
	DCDFLIB::cdfbet(&which, &p, &q, &X, &Y, &inf, &sup, &status, &bound);
	return X;
#else
	throw NotYetImplementedException(HERE);
#endif
      }

      // Debye function of order n: DebyeN(x, n) = n / x^n int_0^x t^n/(\exp(t)-1) dt
      NumericalScalar SpecFunc::Debye(const NumericalScalar x, const UnsignedLong n)
      {
	if ((n == 0) || (n > 20)) throw InvalidArgumentException(HERE) << "Error: cannot compute Debye function of order outside of {1,...,20}";
	if (x < 0.0) return 0.0;
	// The threshold is such that the overall error is less than 1.0e-16
	if (x < 1.0e-8) return 1.0 - n * x / (2.0 * (n + 1.0));
	return debyen(x, static_cast<int>(n)) * n / pow(x, n);
      }

      // Real Faddeeva function: faddeeva(z) = exp(-z^2).erfc(-I*z)
      NumericalComplex SpecFunc::Faddeeva(const NumericalScalar x)
      {
	NumericalScalar xi(x);
	NumericalScalar yi(0.0);
	NumericalScalar u(0.0);
	NumericalScalar v(0.0);
	Bool flag(false);
	WOFZ_F77(&xi, &yi, &u, &v, &flag);
	return NumericalComplex(u, v);
      }

      // Complex Faddeeva function: faddeeva(z) = exp(-z^2).erfc(-I*z)
      NumericalComplex SpecFunc::Faddeeva(const NumericalComplex & z)
      {
	NumericalScalar xi(z.real());
	NumericalScalar yi(z.imag());
	NumericalScalar u(0.0);
	NumericalScalar v(0.0);
	Bool flag(false);
	WOFZ_F77(&xi, &yi, &u, &v, &flag);
	return NumericalComplex(u, v);
      }

      // Gamma function: Gamma(a) = int_0^{\infty} t^(a-1).exp(-t) dt
      NumericalScalar SpecFunc::Gamma(const NumericalScalar a)
      {
	return tgamma(a);
      }

      // Complex gamma function: Gamma(a) = int_0^{\infty} t^(a-1).exp(-t) dt,
      // Computed using Lanczos approximation, using a C++ translation of
      // Paul Godfrey's matlab implementation
      // http://home.att.net/~numericana/answer/info/godfrey.htm#matlabgamma
      NumericalComplex SpecFunc::Gamma(const NumericalComplex & a)
      {
	if (a.imag() == 0.0) return Gamma(a.real());
	const NumericalScalar sqrt2Pi(sqrt(2.0 * M_PI));
	NumericalComplex z(a);
	Bool flip(false);
	if (z.real() < 0.0)
	  {
	    z = -z;
	    flip = true;
	  }
	const UnsignedLong coefficientsSize(11);
	static const NumericalScalar coefficients[coefficientsSize] = {
	  1.000000000000000174663,      5716.400188274341379136,
	  -14815.30426768413909044,     14291.49277657478554025,
	  -6348.160217641458813289,     1301.608286058321874105,
	  -108.1767053514369634679,     2.605696505611755827729,
	  -0.7423452510201416151527e-2, 0.5384136432509564062961e-7,
	  -0.4023533141268236372067e-8};
	const NumericalScalar g(coefficientsSize - 2.0);
	NumericalComplex t(z + g);
	NumericalComplex s(0.0);
	NumericalComplex ss(t - 0.5);
	for (UnsignedLong k = coefficientsSize - 1; k > 0; --k)
	  {
	    s += coefficients[k] / t;
	    t -= 1.0;
	  }
	s += coefficients[0];
	s = log(s * sqrt2Pi) + (z - 0.5) * log(ss) - ss;
	NumericalComplex f(exp(s));
	if (flip) f = -M_PI / (a * f * sin(M_PI * a));
	return f;
      }

      // Natural logarithm of the Gamma function
      NumericalScalar SpecFunc::LnGamma(const NumericalScalar a)
      {
	return lgamma(a);
      }

      // Incomplete Gamma function: GammaInc(a, x) = int_0^x t^(a-1).exp(-t) dt
      NumericalScalar SpecFunc::GammaInc(const NumericalScalar a, const NumericalScalar x)
      {
#ifdef USE_DCDFLIB
	double k(a);
	double t(x);
	double ans;
	double qans;
	int ind(0);
	DCDFLIB::gamma_inc(&k, &t, &ans, &qans, &ind);
	return ans * SpecFunc::Gamma(a);
#else
	NumericalScalar factor(pow(x, a) * exp(-x));
	NumericalScalar denominator(a);
	NumericalScalar term(x / denominator);
	NumericalScalar sum(term);
	NumericalScalar eps;
	do
	  {
	    denominator++;
	    term *= x / denominator;
	    sum += term;
	    eps = fabs(term / sum);
	  }
	while (eps > Precision);
	return factor * sum;
#endif
      }

      // Incomplete Gamma function inverse with respect to x
      NumericalScalar SpecFunc::GammaIncInv(const NumericalScalar a, const NumericalScalar x)
      {
#ifdef USE_DCDFLIB
	double k(a);
	double X;
	double X0(0.0);
	double p(x);
	double q(1.0 - x);
	int ierr;
	DCDFLIB::gamma_inc_inv(&k, &X, &X0, &p, &q, &ierr);
	return X;
#else
	throw NotYetImplementedException(HERE);
#endif
      }

      // Digamma function: Psi(a) = ((dgamma/dx) / gamma)(x)
      // Derivative of a Lanczos approximation of log(Gamma)
      NumericalScalar SpecFunc::Psi(const NumericalScalar x)
      {
	NumericalScalar z(x);
	Bool flip(z < 0.5);
	// For improved accuracy, take the complement of z
	if (flip) z = 1.0 - z;
	NumericalScalar g(607.0 / 128.0);
	const UnsignedLong coefficientsSize(15);
	static const NumericalScalar coefficients[coefficientsSize] = {
	  0.99999999999999709182,    57.156235665862923517,
	  -59.597960355475491248,    14.136097974741747174,
	  -0.49191381609762019978,    0.33994649984811888699e-4,
	  0.46523628927048575665e-4, -0.98374475304879564677e-4,
	  0.15808870322491248884e-3, -0.21026444172410488319e-3,
	  0.21743961811521264320e-3, -0.16431810653676389022e-3,
	  0.84418223983852743293e-4, -0.26190838401581408670e-4,
	  0.36899182659531622704e-5
	};
	NumericalScalar n(0.0);
	NumericalScalar d(0.0);
	for (UnsignedLong k = coefficientsSize - 1; k > 0; --k)
	  {
	    NumericalScalar dz(1.0 / (z + k - 1.0));
	    NumericalScalar dd(coefficients[k] * dz);
	    d += dd;
	    n -= dd * dz;
	  }
	d += coefficients[0];
	NumericalScalar gg(z + g - 0.5);
	NumericalScalar f(log(gg) + (n / d - g / gg));
	if (flip) f -= M_PI / tan(M_PI * z);
	return f;
      }

      // Hypergeometric function of type (1,1): HyperGeom_1_1(p1, q1, x) = sum_{n=0}^{\infty} [prod_{k=0}^{n-1} (p1 + k) / (q1 + k)] * x^n / n!
      NumericalScalar SpecFunc::HyperGeom_1_1(const NumericalScalar p1, const NumericalScalar q1, const NumericalScalar x)
      {	
	NumericalScalar pochhammerP1(p1);
	NumericalScalar pochhammerQ1(q1);
	NumericalScalar factorial(1.0);
	NumericalScalar term(1.0);
	NumericalScalar sum(term);
	NumericalScalar eps;
	do
	  {
	    term *= pochhammerP1 * x / (pochhammerQ1 * factorial);
	    pochhammerP1++;
	    pochhammerQ1++;
	    factorial++;
	    sum += term;
	    eps = fabs(term / sum);
	  }
	while (eps > Precision);
	return sum;
      }

      // Complex hypergeometric function of type (1,1): HyperGeom_1_1(p1, q1, x) = sum_{n=0}^{\infty} [prod_{k=0}^{n-1} (p1 + k) / (q1 + k)] * x^n / n!
      NumericalComplex SpecFunc::HyperGeom_1_1(const NumericalScalar p1, const NumericalScalar q1, const NumericalComplex & x)
      {	
	NumericalComplex pochhammerP1(p1);
	NumericalComplex pochhammerQ1(q1);
	NumericalScalar factorial(1.0);
	NumericalComplex term(1.0);
	NumericalComplex sum(term);
	NumericalComplex eps;
	do
	  {
	    term *= pochhammerP1 * x / (pochhammerQ1 * factorial);
	    pochhammerP1 += 1.0;
	    pochhammerQ1 += 1.0;
	    factorial++;
	    sum += term;
	    eps = term / sum;
	  }
	while (abs(eps) > Precision);
	return sum;
      }

      // Hypergeometric function of type (2,1): HyperGeom_2_1(p1, p2, q1, x) = sum_{n=0}^{\infty} [prod_{k=0}^{n-1} (p1 + k) . (p2 + k) / (q1 + k)] * x^n / n!
      NumericalScalar SpecFunc::HyperGeom_2_1(const NumericalScalar p1, const NumericalScalar p2, const NumericalScalar q1, const NumericalScalar x)
      {	
	NumericalScalar pochhammerP1(p1);
	NumericalScalar pochhammerP2(p2);
	NumericalScalar pochhammerQ1(q1);
	NumericalScalar factorial(1.0);
	NumericalScalar term(1.0);
	NumericalScalar sum(term);
	NumericalScalar eps;
	do
	  {
	    term *= pochhammerP1 * pochhammerP2 * x / (pochhammerQ1 * factorial);
	    pochhammerP1++;
	    pochhammerP2++;
	    pochhammerQ1++;
	    factorial++;
	    sum += term;
	    eps = fabs(term / sum);
	  }
	while (eps > Precision);
	return sum;
      }

      // Hypergeometric function of type (2,2): HyperGeom_2_1(p1, p2, q1, q2, x) = sum_{n=0}^{\infty} [prod_{k=0}^{n-1} (p1 + k) . (p2 + k) / (q1 + k) / (q2 + k)] * x^n / n!
      NumericalScalar SpecFunc::HyperGeom_2_2(const NumericalScalar p1, const NumericalScalar p2, const NumericalScalar q1, const NumericalScalar q2, const NumericalScalar x)
      {	
	NumericalScalar pochhammerP1(p1);
	NumericalScalar pochhammerP2(p2);
	NumericalScalar pochhammerQ1(q1);
	NumericalScalar pochhammerQ2(q1);
	NumericalScalar factorial(1.0);
	NumericalScalar term(1.0);
	NumericalScalar sum(term);
	NumericalScalar eps;
	do
	  {
	    term *= pochhammerP1 * pochhammerP2 * x / (pochhammerQ1 * pochhammerQ2 * factorial);
	    pochhammerP1++;
	    pochhammerP2++;
	    pochhammerQ1++;
	    pochhammerQ2++;
	    factorial++;
	    sum += term;
	    eps = fabs(term / sum);
	  }
	while (eps > Precision);
	return sum;
      }

      // Erf function Erf(x) = 2 / sqrt(Pi) . int_0^x exp(-t^2) dt
      NumericalScalar SpecFunc::Erf(const NumericalScalar x)
      {
	return erf(x);
      }

      // Erfc function ErfC(x) = 1 - Erf(x)
      NumericalScalar SpecFunc::ErfC(const NumericalScalar x)
      {
	return erfc(x);
      }

      // Inverse of the Erf function
      NumericalScalar SpecFunc::ErfInv(const NumericalScalar x)
      {
	//#ifdef USE_DCDFLIB
	//	NumericalScalar p(0.5 * (x + 1.0));
	//	NumericalScalar q(1.0 - p);
	//	return M_SQRT1_2 * DCDFLIB::dinvnr((double*)&p, (double*)&q);
	//#else
	NumericalScalar p(0.5 * (x + 1.0));
	static const NumericalScalar a[6] = {
	  -3.969683028665376e+01,  2.209460984245205e+02,
	  -2.759285104469687e+02,  1.383577518672690e+02,
	  -3.066479806614716e+01,  2.506628277459239e+00
	};
	static const NumericalScalar b[5] = {
	  -5.447609879822406e+01,  1.615858368580409e+02,
	  -1.556989798598866e+02,  6.680131188771972e+01,
	  -1.328068155288572e+01
	};
	static const NumericalScalar c[6] = {
	  -7.784894002430293e-03, -3.223964580411365e-01,
	  -2.400758277161838e+00, -2.549732539343734e+00,
	  4.374664141464968e+00,  2.938163982698783e+00
	};
	static const NumericalScalar d[4] = {
	  7.784695709041462e-03,  3.224671290700398e-01,
	  2.445134137142996e+00,  3.754408661907416e+00
	};
	NumericalScalar q;
	NumericalScalar t;
	NumericalScalar u;
	q = std::min(p, 1.0 - p);
	if (q > 0.02425) {
	  /* Rational approximation for central region. */
	  u = q - 0.5;
	  t = u * u;
	  u = u * (((((a[0] * t + a[1]) * t + a[2]) * t + a[3]) * t + a[4]) * t + a[5])
	    /(((((b[0] * t + b[1]) * t + b[2]) * t + b[3]) * t + b[4]) * t + 1.0);
	} else {
	  /* Rational approximation for tail region. */
	  t = sqrt(-2.0 * log(q));
	  u = (((((c[0] * t + c[1]) * t + c[2]) * t + c[3]) * t + c[4]) * t + c[5])
	    /((((d[0] * t + d[1]) * t + d[2]) * t + d[3]) *t + 1.0);
	}
	/* The relative error of the approximation has absolute value less
	   than 1.15e-9.  One iteration of Halley's rational method (third
	   order) gives full machine precision... */
	t = 0.5 + 0.5 * Erf(u * M_SQRT1_2) - q;    /* f(u) = error */
	// 2.50662827463100050241576528481 = sqrt(2.pi)
	t = t * 2.50662827463100050241576528481 * exp(0.5 * u * u);   /* f(u)/df(u) */
	u = u - t / (1.0 + 0.5 * u * t);     /* Halley's method */
	return (p > 0.5 ? -M_SQRT1_2 * u : M_SQRT1_2 * u);
	//#endif
      }

      /* Evaluation of the principal barnch of Lambert W function.
	 Based on formulas exposed in:
	 Robert M. Corless, G. H. Gonnet, D. E. G. Hare, D. J. Jeffrey, and D. E. Knuth, "On the Lambert W Function", Advances in Computational Mathematics, volume 5, 1996, pp. 329--359
      */
      NumericalScalar SpecFunc::LambertW(const NumericalScalar x, const Bool principal)
      {
	NumericalScalar w(-1.0);
	// -0.36787944117144232159552377016146086 = -1 / exp(1)
	if (x <= -0.3678794411714423215955238) return w;
	// Principal branch, defined over [-1/e,+inf], LambertW >= -1
	if (principal)
	  {
	    if (x == 0) return 0.0;
	    if(x < 6.46)
	      {
		w = x * (3.0 + 4.0 * x) / (3.0 + x * (7.0 + 2.5 * x));
	      }
	    // Large argument, use asymptotic expansion, formula 4.18
	    else
	      {
		const NumericalScalar t1(log(x));
		w = t1 - log(t1);
	      }
	  }
	// Second real branch, defined over [-1/e, 0[, LambertW <= -1
	else
	  {
	    if (x >= 0.0) return -INFINITY;
	    if (x < -0.1)
	      {
		w = -2.0;
	      }
	    else
	      {
		const NumericalScalar t1(log(-x));
		w = t1 - log(-t1);
	      }
	  }
	// Halley's iteration
	for (UnsignedLong i = 0; i < 3; ++i)
	  {
	    const NumericalScalar expW(exp(w));
	    const NumericalScalar numerator(w * expW - x);
	    const NumericalScalar dw(numerator / (expW * (w + 1.0) - 0.5 * (w + 2.0) * numerator / (w + 1.0)));
	    w -= dw;
	    if (fabs(dw) < Precision) break; 
	  }
	return w;
      }

    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
