//                                               -*- C++ -*-
/**
 *  @file  Wilks.cxx
 *  @brief Wilks is a generic view of Wilks methods for computing
 * probabilities and related quantities by sampling and estimation
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu 26 jun 2008) $
 *  Id:      $Id: Wilks.cxx 862 2008-06-26 11:50:17Z dutka $
 */
#include <cmath>
#include "Wilks.hxx"
#include "NumericalPoint.hxx"
#include "DistFunc.hxx"

namespace OpenTURNS
{

  namespace Uncertainty 
  {

    namespace Algorithm
    {

      typedef Base::Type::NumericalPoint NumericalPoint;
      typedef Distribution::DistFunc     DistFunc;

      /*
       * @class Wilks
       */

      /* Sample size computation */
      UnsignedLong Wilks::ComputeSampleSize(const NumericalScalar quantileLevel,
					    const NumericalScalar confidenceLevel,
					    const UnsignedLong marginIndex)
	throw(InvalidArgumentException)
      {
	if ((quantileLevel <= 0.0) || (quantileLevel >= 1.0)) throw InvalidArgumentException(HERE) << "Error: quantile level must be in ]0, 1[, here quantile level=" << quantileLevel;
	if ((confidenceLevel <= 0.0) || (confidenceLevel >= 1.0)) throw InvalidArgumentException(HERE) << "Error: confidence level must be in ]0, 1[, here confidence level=" << confidenceLevel;
	// Initial guess based on asymptotic normality of the empirical quantile
	NumericalScalar x(DistFunc::qNormal(confidenceLevel));
	NumericalScalar x2(x * x);
	UnsignedLong n((UnsignedLong)(ceil((quantileLevel * x2 + 2.0 * marginIndex + sqrt(quantileLevel * x2 * (quantileLevel * x2 + 4.0 * marginIndex))) / (2.0 * (1.0 - quantileLevel)))));
	// Initialize cQuantileLevel to cQuantileLevel[i] = Binomial(n, n - marginIndex + i) * quantileLevel ^ (n - marginIndex + i) * (1 - quantileLevel) ^ (marginIndex - i)
	// For i = marginIndex, cQuantileLevel[i] = quantileLevel ^ n
	NumericalPoint cQuantileLevel(marginIndex + 1, pow(quantileLevel, n));
	// Descending recursion n! / (m - i)! / (n - m + i)! = n! / (m - (i - 1))! / (n - m + (i - 1))! * (m - (i - 1)) / (n - m + i)
	// -> Binomial(n, n - m + i - 1) = Binomial(n, n - m + i) * (n - m + i) / (m - i)
	for (UnsignedLong i = marginIndex; i > 0; i--)
	  {
	    cQuantileLevel[i - 1] = cQuantileLevel[i] * (1.0 - quantileLevel) / quantileLevel * (n - marginIndex + i) / (marginIndex - i + 1.0);
	  }
	NumericalScalar cumulQuantileLevel;
        do
	  {
	    cumulQuantileLevel = 0.0;
	    n++;
	    // Update the terms in the sum
	    // Binomial(n + 1, n + 1 - m + j) * quantileLevel ^ (n + 1 - m + j) * (1 - quantileLevel) ^ (m - j)= Binomial(n, n - m + j) * quantileLevel ^ (n - m + j) * (1 - quantileLevel) ^ (m - j) * quantileLevel * (n + 1) / (n + 1 - m + j)
	    for (UnsignedLong j = 0; j <= marginIndex; j++)
	      {
		cQuantileLevel[j] *= (n * quantileLevel) / (n - marginIndex + j);
		cumulQuantileLevel += cQuantileLevel[j];
	      }
	  }
	while (cumulQuantileLevel > 1.0 - confidenceLevel);
	return n;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
