//                                               -*- C++ -*-
/**
 *  @file  DirectionalSampling.cxx
 *  @brief DirectionalSampling is an implementation of the directional
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-22 11:34:11 +0200 (lun 22 sep 2008) $
 *  Id:      $Id: DirectionalSampling.cxx 941 2008-09-22 09:34:11Z dutka $
 */
#include "DirectionalSampling.hxx"
#include "CorrelationMatrix.hxx"
#include "LinearNumericalMathFunction.hxx"
#include "NumericalMathFunction.hxx"
#include "Matrix.hxx"
#include "ComparisonOperatorImplementation.hxx"

namespace OpenTURNS
{

  namespace Uncertainty 
  {

    namespace Algorithm
    {

      typedef Base::Stat::CorrelationMatrix             CorrelationMatrix;
      typedef Base::Func::LinearNumericalMathFunction   LinearNumericalMathFunction;
      typedef Base::Func::NumericalMathFunction         NumericalMathFunction;
      typedef Base::Type::Matrix                        Matrix;
      typedef Base::Type::NumericalPoint                NumericalPoint;

      /*
       * @class DirectionalSampling
       */

      CLASSNAMEINIT(DirectionalSampling);

      /* Constructor with parameters */
      DirectionalSampling::DirectionalSampling(const Simulation::Event & event)
	throw(InvalidArgumentException):
	Simulation(event),
	standardEvent_(StandardEvent(event)),
	standardFunction_(standardEvent_.getImplementation()->getFunction()),
	inputDistribution_(standardEvent_.getImplementation()->getAntecedent()->getDistribution().getImplementation()),
	rootStrategy_(),
	samplingStrategy_(inputDistribution_->getDimension())
      {
	// Nothing to do
      }

      /* Constructor with parameters */
      DirectionalSampling::DirectionalSampling(const Simulation::Event & event,
					       const RootStrategy & rootStrategy,
					       const SamplingStrategy & samplingStrategy)
	throw(InvalidArgumentException):
	Simulation(event),
	standardEvent_(StandardEvent(event)),
	standardFunction_(standardEvent_.getImplementation()->getFunction()),
	inputDistribution_(standardEvent_.getImplementation()->getAntecedent()->getDistribution().getImplementation()),
	rootStrategy_(rootStrategy),
	samplingStrategy_(samplingStrategy)
      {
	// To force the sampling strategy to have the correct dimension
	samplingStrategy_.setDimension(inputDistribution_->getDimension());
      }

      /* Virtual constructor */
      DirectionalSampling * DirectionalSampling::clone() const
      {
	return new DirectionalSampling(*this);
      }

      /* Compute the contribution of a direction to the probability given the roots of the performance function along the direction
         Starting from the origin of the direction, if we suppose that the first root is a transition form the safe space to the
         failure space, the contribution of the direction is:
         dP = sum_{k=0}^{n-1}(-1)^k(1-F(x_k)), where (x_k)_{k=0,...,n-1} are the n roots along the direction
         This expression can be rewritten in a more numerically stable form:
         dP = -sum_{k=0}^{n-1}(-1)^kF(x_k) + e(n), with e(n) = 0 if n is even and e(n) = 1 if n is odd.
         If the origin is already in the failure space, the contribution of the direction is the tail to 1.0 of the preceeding expression.
      */
      NumericalScalar DirectionalSampling::computeContribution(const NumericalScalarCollection & roots)
      {
	NumericalScalar sign(-1.);
	NumericalScalar estimate(0.0);
	UnsignedLong size(roots.getSize());
	for (UnsignedLong indexRoot = 0; indexRoot < size; indexRoot++)
	  {
	    NumericalScalar currentRoot(roots[indexRoot]);
	    estimate += sign * inputDistribution_->computeRadialDistributionCDF(currentRoot);
	    sign = -sign;
	  }
	if (size % 2 != 0) estimate += 1.0;
	// Is the origin in the failure space?
	// Here, we know that the getOriginValue() method will not throw an exception, as we already called the solve() method
        // of the root strategy, which in turn initialized the computation of the origin value.
	if (standardEvent_.getOperator().operator()(rootStrategy_.getOriginValue(), standardEvent_.getThreshold())) estimate = 1.0 - estimate;
	return estimate;
      }

      /* Compute the contribution of a set of directions direction to the probability */
      NumericalScalar DirectionalSampling::computeTotalContribution(const NumericalSample & directionSample)
      {
	NumericalScalar contribution(0.0);
	UnsignedLong sampleSize(directionSample.getSize());
	UnsignedLong dimension(directionSample.getDimension());
	Matrix linear(dimension, 1);
	// For each direction
	for (UnsignedLong indexDirection = 0; indexDirection < sampleSize; indexDirection++)
	  {
	    NumericalPoint direction(directionSample[indexDirection]);
	    // First compute the roots along this direction
	    // 1. Build the scalar function along the direction
	    // 1.1 Build the linear function along the direction
	    for (UnsignedLong indexComponent = 0; indexComponent < dimension; indexComponent++)
	      {
		linear(indexComponent, 0) = direction[indexComponent];
	      }
	    LinearNumericalMathFunction ray(NumericalPoint(1, 0.0), NumericalPoint(dimension, 0.0), linear);
	    // 1.2 Build the function along the ray
	    NumericalMathFunction functionAlongRay(standardFunction_, ray);
	    // 2. Solve the function along the ray
	    NumericalScalarCollection roots(rootStrategy_.solve(functionAlongRay, standardEvent_.getThreshold()));
	    // Second, sum up the contribution of this direction
	    contribution += computeContribution(roots);
	  }
	return contribution / sampleSize;
      }

      /* Compute the block sample */
      DirectionalSampling::NumericalSample DirectionalSampling::computeBlockSample()
      {
	UnsignedLong size(getBlockSize());
	NumericalSample blockSample(size, 1);
	// For each entry of the block sample
	for (UnsignedLong index = 0; index < size; index++)
	  {
	    NumericalSample directionSample(samplingStrategy_.generate());
	    // Compute the contribution of the sub-sample computed according to the sampling strategy
	    blockSample[index] = NumericalPoint(1, computeTotalContribution(directionSample));
	  }
	return blockSample;
      }

      /* Root strategy accessor */
      void DirectionalSampling::setRootStrategy(const RootStrategy & rootStrategy)
      {
	rootStrategy_ = rootStrategy;
      }

      RootStrategy DirectionalSampling::getRootStrategy() const
      {
	return rootStrategy_;
      }

      /* Sampling strategy */
      void DirectionalSampling::setSamplingStrategy(const SamplingStrategy & samplingStrategy)
      {
	if (samplingStrategy.getDimension() != getEvent().getImplementation()->getAntecedent()->getDimension()) throw InvalidArgumentException(HERE) << "Error: the sampling strategy dimension is not compatible with the antecedent dimension";
	samplingStrategy_ = samplingStrategy;
      }

      SamplingStrategy DirectionalSampling::getSamplingStrategy() const
      {
	return samplingStrategy_;
      }

      /* String converter */
      String DirectionalSampling::str() const
      {
	OSS oss;
	oss << "class=" << DirectionalSampling::GetClassName()
	    << " rootStrategy=" << rootStrategy_.str()
	    << " samplingStrategy=" << samplingStrategy_.str()
	    << " derived from " << Simulation::str();
	return oss;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
