//                                               -*- C++ -*-
/**
 *  @file  InverseNatafEllipticalDistributionEvaluation.cxx
 *  @brief Class for the InverseNataf transformation evaluation for elliptical
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: InverseNatafEllipticalDistributionEvaluation.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include "OT.hxx"
#include "InverseNatafEllipticalDistributionEvaluation.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {


      /*
       * @class InverseNatafEllipticalDistributionEvaluation
       *
       * This class offers an interface for the InverseNataf function for elliptical distributions
       */

      CLASSNAMEINIT(InverseNatafEllipticalDistributionEvaluation);

      /* Parameter constructor */
      InverseNatafEllipticalDistributionEvaluation::InverseNatafEllipticalDistributionEvaluation(const NumericalPoint & mean,
												 const SquareMatrix & cholesky):
	LinearNumericalMathEvaluationImplementation(
						    NumericalPoint(mean.getDimension(), 0.0),
						    mean,
						    cholesky.transpose()
						    )
      {
	// Nothing to do
      }

      /* Destructor */
      InverseNatafEllipticalDistributionEvaluation::~InverseNatafEllipticalDistributionEvaluation()
      {
	// Nothing to do
      }

      /* String converter */
      String InverseNatafEllipticalDistributionEvaluation::str() const
      {
	OSS oss;
	oss << "class=" << InverseNatafEllipticalDistributionEvaluation::GetClassName()
            << " mean=" << getCenter()
            << " cholesky=" << getLinear().transpose();
                   
	return oss;
      }

      /* Gradient according to the marginal parameters.
       * For the InverseNataf transformation based on elliptical
       * distributions, the marginal parameters \theta are the mean
       * vector \mu and the marginal standard deviations \sigma,
       * which are the diagonal terms of the Cholesky factor L
       * of the covariance matrix: \theta = (\mu, \sigma)
       * The expression of the inverse transformation is:
       *
       * T^(-1)_i(U) = mu_i + \sum_{j=1}^i L_{ij}U_j
       *
       * thus we have:
       *
       * dT^(-1)_i/dmu_j = 1 for j=i,
       *                 = 0 otherwise
       * dT^(-1)_i/dsigma_j = U_i for j=i,
       *                    = 0 otherwise
       *
       * The (i,j)th element of the gradient is dT^(-1)_j/dtheta_i.
       *
       */

      InverseNatafEllipticalDistributionEvaluation::Matrix InverseNatafEllipticalDistributionEvaluation::parametersGradient(const NumericalPoint & in) const
      {
	UnsignedLong inputDimension(getInputNumericalPointDimension());
	Matrix result(2 * inputDimension, inputDimension);
	// dT_j/dmu_i
	for (UnsignedLong i = 0; i < inputDimension; ++i)
	  {
	    result(i, i) = 1.0;
	  }
	// dTj/dsigma_j
	for (UnsignedLong i = 0; i < inputDimension; ++i)
	  {
	    result(inputDimension + i, i) = in[i];
	  }
	return result;
      }
	  
    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
