//                                               -*- C++ -*-
/**
 *  @file  MarginalTransformationEvaluation.cxx
 *  @brief Class for the Nataf transformation evaluation for elliptical
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: MarginalTransformationEvaluation.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include "MarginalTransformationEvaluation.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {



      CLASSNAMEINIT(MarginalTransformationEvaluation);

      /* Parameter constructor */
      MarginalTransformationEvaluation::MarginalTransformationEvaluation(const DistributionCollection & distributionCollection):
	NumericalMathEvaluationImplementation(),
	distributionCollection_(distributionCollection)
      {
	Description description;
	const UnsignedLong size(distributionCollection.getSize());
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    OSS oss;
	    oss << "x" << i;
	    description.add(oss);
	  }
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    OSS oss;
	    oss << "y" << i;
	    description.add(oss);
	  }
	setDescription(description);
      }

      /* Virtual constructor */
      MarginalTransformationEvaluation * MarginalTransformationEvaluation::clone() const
      {
	return new MarginalTransformationEvaluation(*this);
      }

      /* Evaluation */
      MarginalTransformationEvaluation::NumericalPoint MarginalTransformationEvaluation::operator () (const NumericalPoint & in) const
	throw (InvalidArgumentException, InternalException)
      {
	const UnsignedLong dimension(getOutputNumericalPointDimension());
	NumericalPoint result(dimension);
	// Apply CDF over the components
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    result[i] = distributionCollection_[i].computeCDF(NumericalPoint(1, in[i]));
	  }
	return result;
      }

      /* Gradient according to the marginal parameters. */
      MarginalTransformationEvaluation::Matrix MarginalTransformationEvaluation::parametersGradient(const NumericalPoint & in) const
      {
	const NumericalPoint parameters(getParameters());
	const UnsignedLong parametersDimension(parameters.getDimension());
	const UnsignedLong inputDimension(getInputNumericalPointDimension());
	Matrix result(parametersDimension, inputDimension);
	UnsignedLong rowIndex(0);
	for (UnsignedLong j = 0; j < inputDimension; ++j)
	  {
	    const NumericalPoint marginalCDFGradient(distributionCollection_[j].computeCDFGradient(NumericalPoint(1, in[j])));
	    const UnsignedLong marginalParametersDimension(marginalCDFGradient.getDimension());
	    for (UnsignedLong i = 0; i < marginalParametersDimension; ++i)
	      {
		result(rowIndex, j) = marginalCDFGradient[i];
		rowIndex++;
	      }
	  }
	return result;
      }

      /* Accessor for input point dimension */
      UnsignedLong MarginalTransformationEvaluation::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return distributionCollection_.getSize();
      }

      /* Accessor for output point dimension */
      UnsignedLong MarginalTransformationEvaluation::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return distributionCollection_.getSize();
      }

      /* String converter */
      String MarginalTransformationEvaluation::str() const
      {
	OSS oss;
	oss << "class=" << MarginalTransformationEvaluation::GetClassName()
	    << " description=" << getDescription()
	    << " marginals=" << distributionCollection_;                   
	return oss;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

