//                                               -*- C++ -*-
/**
 *  @file  InverseMarginalTransformationHessian.cxx
 *  @brief Class for the Nataf transformationHessian evaluation for elliptical
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: InverseMarginalTransformationHessian.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include "InverseMarginalTransformationHessian.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {



      CLASSNAMEINIT(InverseMarginalTransformationHessian);

      /* Parameter constructor */
      InverseMarginalTransformationHessian::InverseMarginalTransformationHessian(const DistributionCollection & distributionCollection):
	NumericalMathHessianImplementation(),
	distributionCollection_(distributionCollection)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      InverseMarginalTransformationHessian * InverseMarginalTransformationHessian::clone() const
      {
	return new InverseMarginalTransformationHessian(*this);
      }

      /* Hessian */
      InverseMarginalTransformationHessian::SymmetricTensor InverseMarginalTransformationHessian::hessian(const NumericalPoint & in) const
	throw (InvalidArgumentException, InternalException)
      {
	// As the marginal transformation is a function from Rn to Rn, the hessian is a cubical symmetric tensor
	UnsignedLong dimension(getInputNumericalPointDimension());
	SymmetricTensor result(dimension, dimension);
	// Apply quantile second derivative, i.e. -DDF(Quantile) / PDF(Quantile)^3 over the components
	// GoF=I -> F'.G'oF=1 -> G'=1/F'oG -> G''=-G'.F''oG/(F'oG)^2 -> G''=-F''oG/(F'oG)^3
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    NumericalPoint q(distributionCollection_[i].computeQuantile(in[i]));
	    NumericalScalar denominator(distributionCollection_[i].computePDF(q));
	    result(i, i, i) = -distributionCollection_[i].computeDDF(q)[0] / (denominator * denominator * denominator);
	  }
	return result;
      }
	  
      /* Accessor for input point dimension */
      UnsignedLong InverseMarginalTransformationHessian::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	return distributionCollection_.getSize();
      }

      /* Accessor for output point dimension */
      UnsignedLong InverseMarginalTransformationHessian::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	return distributionCollection_.getSize();
      }

      /* String converter */
      String InverseMarginalTransformationHessian::str() const
      {
	OSS oss;
	oss << "class=" << InverseMarginalTransformationHessian::GetClassName();                   
	return oss;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

