//                                               -*- C++ -*-
/**
 *  @file  Tensor.hxx
 *  @brief Tensor implements the classical mathematical tensor
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: Tensor.hxx 995 2008-10-31 10:52:04Z dutka $
 */
#ifndef OPENTURNS_TENSOR_HXX
#define OPENTURNS_TENSOR_HXX

#include "TensorImplementation.hxx"
#include "TypedInterfaceObject.hxx"


namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      class TensorImplementation;

      /**
       * @class Tensor
       *
       * Tensor implements the classical mathematical Tensor
       */

      class Tensor :
	public Common::TypedInterfaceObject<TensorImplementation>
      
      {
	CLASSNAME;
	
      public:
	
	typedef Common::InvalidDimensionException                                  InvalidDimensionException;
	typedef Common::InvalidArgumentException                                   InvalidArgumentException;
	typedef Common::TypedInterfaceObject<TensorImplementation>::Implementation Implementation;
	
	/** Default constructor */
	Tensor();

	/** Constructor with size (rowDim, colDim, sheetDim) */
	Tensor(const UnsignedLong rowDim,
	       const UnsignedLong colDim,
	       const UnsignedLong sheetDim);
	
	/** Constructor from range of external collection */
	template <class InputIterator>
	Tensor(const UnsignedLong rowDim,
	       const UnsignedLong colDim,
	       const UnsignedLong sheetDim,
	       InputIterator first,
	       InputIterator last);
	       
	/** Constructor from external collection */
	Tensor(const UnsignedLong rowDim,
	       const UnsignedLong colDim,
	       const UnsignedLong sheetDim,
	       const Collection<NumericalScalar> & elementsValues);
	
	/** String converter */
	virtual String str() const;

#ifndef SWIG	
	/** Operator () gives access to the elements of the tensor (to modify these elements) */
	/** The element of the tensor is designated by its row number i, its column number j and its sheet number k */
	NumericalScalar & operator () (const UnsignedLong i,
				       const UnsignedLong j,
				       const UnsignedLong k) throw(InvalidDimensionException);
	
	/** Operator () gives access to the elements of the tensor (read only) */
	/** The element of the tensor is designated by its row number i, its column number j and its sheet number k */
	const NumericalScalar & operator () (const UnsignedLong i,
					     const UnsignedLong j,
					     const UnsignedLong k) const throw(InvalidDimensionException);
#endif

	/** getSheet returns the sheet specified by its sheet number k */
	Matrix getSheet(const UnsignedLong k) const throw(InvalidDimensionException);
	
	/** setSheet sets matrix m as the sheet specified by its sheet number k  */
	void setSheet(const UnsignedLong k,
		      const Matrix & m) throw(InvalidDimensionException);
	
	/** Get the dimensions of the tensor */
	/** Number of rows */
	const UnsignedLong getNbRows() const ;
	/** Number of columns */
	const UnsignedLong getNbColumns() const ;
	/** Number of sheets */
	const UnsignedLong getNbSheets() const ;
	
	/** Comparison operators */
	Bool operator == (const Tensor & rhs) const ;
	
	/** Empty returns true if there is no element in the tensor */
	const Bool isEmpty() const;

      protected:
        
	/** Constructor with implementation */
	Tensor(const Implementation & i);

      }; /* class Tensor */



      /** Constructor with size (rowDim, colDim, sheetDim) */
      template <class InputIterator>
      Tensor::Tensor(const UnsignedLong rowDim,
		     const UnsignedLong colDim,
		     const UnsignedLong sheetDim,
		     InputIterator first,
		     InputIterator last)
	: Common::TypedInterfaceObject<TensorImplementation>(new TensorImplementation(rowDim, colDim, sheetDim, first, last))
      {
	// nothing to do
      }

    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_MATRIX_HXX */
