//                                               -*- C++ -*-
/**
 *  @file  NumericalPoint.cxx
 *  @brief NumericalPoint implements the classical mathematical point
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: NumericalPoint.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include <cmath>
#include "NumericalPoint.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      typedef Common::OutOfBoundException OutOfBoundException;

      CLASSNAMEINIT(NumericalPoint);

      /* Default constructor */
      NumericalPoint::NumericalPoint()
	: Common::TypedCollectionInterfaceObject<NumericalPointImplementation>(new NumericalPointImplementation)
      {
        // Nothing to do
      }

      /* Constructor with size */
      NumericalPoint::NumericalPoint(const UnsignedLong size,
				     const NumericalScalar value)
	: Common::TypedCollectionInterfaceObject<NumericalPointImplementation>(new NumericalPointImplementation(size, value))
      {
        // Nothing to do
      }

      /* Constructor from implementation */
      NumericalPoint::NumericalPoint(const NumericalPointImplementation & implementation)
	: Common::TypedCollectionInterfaceObject<NumericalPointImplementation>(implementation.clone())
      {
	// Nothing to do
      }


      /* Constructor from implementation */
      NumericalPoint::NumericalPoint(NumericalPointImplementation * p_implementation, Derived )
	: Common::TypedCollectionInterfaceObject<NumericalPointImplementation>(p_implementation)
      {
	// Nothing to do
      }


      /* Method __eq__() is for Python */
      Bool NumericalPoint::__eq__(const NumericalPoint & rhs) const
      {
	return *this == rhs;
      }

      /* Method __ne__() is for Python */
      Bool NumericalPoint::__ne__(const NumericalPoint & rhs) const
      {
	return *this != rhs;
      }

      /* Collection accessor */
      const Type::Collection<NumericalScalar> & NumericalPoint::getCollection() const
      {
        return getImplementation()->getCollection();
      }


      /* String converter */
      String NumericalPoint::str() const
      {
	return OSS() << "class=" << NumericalPoint::GetClassName()
		     << " name=" << getName()
		     << " dimension=" << getDimension()
		     << " implementation=" << getImplementation()->str();
      }




//       /* Description Accessor */
//       void NumericalPoint::setDescription(const Description & description)
//       {
// 	copyOnWrite();
// 	getImplementation()->setDescription(description);
//       }


//       /* Description Accessor */
//       Description NumericalPoint::getDescription() const
//       {
// 	return getImplementation()->getDescription();
//       }


      /*
       * @fn std::ostream & operator <<(std::ostream & os, const NumericalPoint & obj)
       * @brief Output stream converter
       * @param os A STL output stream object
       * @param obj The object read by \em os
       * @return A reference to \em os
       */
      std::ostream & operator <<(std::ostream & os, const NumericalPoint & obj)
      {
        return os << obj.str();
      }


      /* Addition operator */
      NumericalPoint NumericalPoint::operator +(const NumericalPoint & rhs) const
	throw (InvalidArgumentException)
      {
	return *getImplementation() + *(rhs.getImplementation());
      }


      /* In-place addition operator */
      NumericalPoint & NumericalPoint::operator +=(const NumericalPoint & other)
 	throw (InvalidArgumentException)
      {
	copyOnWrite();
	*getImplementation() += *(other.getImplementation());
	return *this;
      }
      


      /* Substraction operator */
      NumericalPoint NumericalPoint::operator -(const NumericalPoint & rhs) const
 	throw (InvalidArgumentException)
      {
	return *getImplementation() - *(rhs.getImplementation());
      }



      /* In-place substraction operator */
      NumericalPoint & NumericalPoint::operator -=(const NumericalPoint & other)
 	throw (InvalidArgumentException)
      {
	copyOnWrite();
	*getImplementation() -= *(other.getImplementation());
	return *this;
      }
      


      /* Product operator */
      NumericalPoint NumericalPoint::operator *(const NumericalScalar scalar) const
      {
	return *getImplementation() * scalar;
      }

      /*  In-place product operator */
      NumericalPoint & NumericalPoint::operator *=(const NumericalScalar scalar)
      {
	copyOnWrite();
	*getImplementation() *= scalar;
	return *this;
      }



      /* Product operator */
      NumericalPoint operator *(const NumericalScalar scalar,
				const NumericalPoint & point)
      {
	return point * scalar;
      }



      /* Dot product operator */
      NumericalScalar NumericalPoint::dot(const NumericalPoint & lhs,
					  const NumericalPoint & rhs)
	throw (InvalidArgumentException)
      {
	return NumericalPointImplementation::dot( *(lhs.getImplementation()), *(rhs.getImplementation()));
      }


      /* Comparison operator */
      Bool operator ==(const NumericalPoint & lhs,
		       const NumericalPoint & rhs)
      {
	return *(lhs.getImplementation()) == *(rhs.getImplementation());
      }

      /* Ordering operator */
      Bool operator <(const NumericalPoint & lhs,
		      const NumericalPoint & rhs)
      {
	return *(lhs.getImplementation()) < *(rhs.getImplementation());
      }



      /*  Norm */
      NumericalScalar NumericalPoint::norm() const
      {
	return sqrt( norm2() );
      }
	

      /*  Norm^2 */
      NumericalScalar NumericalPoint::norm2() const
      {
	return NumericalPointImplementation::dot( *getImplementation(), *getImplementation());
      }
	


    } /* namespace Type */
  } /* namespace Base */
} /* namespace OpenTURNS */
