//                                               -*- C++ -*-
/**
 *  @file  Last.cxx
 *  @brief 
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: Last.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include "Last.hxx"
#include "NumericalSampleImplementation.hxx"

namespace OpenTURNS
{

  namespace Base 
  {

    namespace Stat
    {

      /**
       * @class Last
       */

      CLASSNAMEINIT(Last);

      const UnsignedLong Last::DefaultMaximumSize;

      /* Constructor with parameters */
      Last::Last()
	: HistoryStrategyImplementation(),
	  maximumSize_(DefaultMaximumSize),
	  index_(0),
	  hasWrapped_(false)
      {
	// Nothing to do
      }

      /* Constructor with parameters */
      Last::Last(const UnsignedLong maximumSize)
	: HistoryStrategyImplementation(),
	  maximumSize_(maximumSize),
	  index_(0),
	  hasWrapped_(false)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      Last * Last::clone() const
      {
	return new Last(*this);
      }

      /* Store the point according to the strategy */
      void Last::store(const NumericalPoint & point)
      {
	if (!isInitialized_)
	  {
	    sample_ = NumericalSample(maximumSize_, point.getDimension());
	    isInitialized_ = true;
	  }
	sample_[index_] = point;
	++index_;
	if (index_ == maximumSize_)
	  {
	    index_ = 0;
	    hasWrapped_ = true;
	  }
      }

      /* Sample accessor */
      NumericalSample Last::getSample() const
      {
	// If nothing has been stored
	if (!isInitialized_) return sample_;
	// If we don't have exhausted the maximum size, return just the stored data
	if (!hasWrapped_)
	  {
	    NumericalSample outSample(index_, sample_.getDimension());
	    for (UnsignedLong i = 0; i < index_; ++i)
	      {
		// Fill the outSample component-wise in order to avoid memory comsumption due to the creation of Description objects
		for (UnsignedLong j = 0; j < sample_.getDimension(); ++j)
		  {
		    outSample[i][j] = sample_[i][j];
		  }
	      }
	    return outSample;
	  }
	// Perform a permutation of the elements: 
	NumericalSample outSample(maximumSize_, sample_.getDimension());
	for (UnsignedLong i = 0; i < maximumSize_; ++i)
	  {
	    UnsignedLong circularIndex((index_ + i) % maximumSize_);
	    // Fill the outSample component-wise in order to avoid memory comsumption due to the creation of Description objects
	    for (UnsignedLong j = 0; j < sample_.getDimension(); ++j)
	      {
		outSample[i][j] = sample_[circularIndex][j];
	      }
	  }
	return outSample;
      }

      /* MaximumSize accessor */
      UnsignedLong Last::getMaximumSize() const
      {
	return maximumSize_;
      }

      /* Index accessor */
      UnsignedLong Last::getIndex() const
      {
	return index_;
      }

      /* Wrapped flag accessor */
      Bool Last::getHasWrapped() const
      {
	return hasWrapped_;
      }

      /* String converter */
      String Last::str() const
      {
	OSS oss;
	oss << "class=" << Last::GetClassName();
	return oss;
      }

    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */
