//                                               -*- C++ -*-
/**
 * @file  CorrelationAnalysis.cxx
 * @brief CorrelationAnalysis implements the sensitivity analysis methods based on correlation coefficients
 *
 * (C) Copyright 2005-2006 EADS
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * @author $LastChangedBy: dutka $
 * @date   $LastChangedDate: 2008-09-13 22:37:56 +0200 (sam 13 sep 2008) $
 */

#include <cmath>
#include <fstream>
#include "CorrelationAnalysis.hxx"
#include "Exception.hxx"
#include "LinearModelFactory.hxx"
#include "LinearModel.hxx"

namespace OpenTURNS
{
  namespace Base
  {
    namespace Stat
    {

      typedef Common::InvalidArgumentException        InvalidArgumentException;
      typedef Common::InvalidDimensionException       InvalidDimensionException;

      /* Compute the Pearson correlation coefficient between the component number index of the input sample and the 1D outputSample */
      NumericalScalar CorrelationAnalysis::PearsonCorrelation(const NumericalSample & inputSample,
							      const NumericalSample & outputSample,
							      const UnsignedLong index)
      {
	if (index >= inputSample.getDimension()) throw InvalidArgumentException(HERE) << "Error: given index out of bound";
	if (outputSample.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: output sample must be 1D";
	if (inputSample.getSize() != outputSample.getSize()) throw InvalidArgumentException(HERE) << "Error: input and output samples must have the same size";
	UnsignedLong size(inputSample.getSize());
	NumericalSample pairedSample(size, 2);
	for (UnsignedLong i = 0; i < size; i++)
	  {
	    pairedSample[i][0] = inputSample[i][index];
	    pairedSample[i][1] = outputSample[i][0];
	  }
	return pairedSample.computePearsonCorrelation()(0, 1);
      }

      /* Compute the Spearman correlation coefficient between the component number index of the input sample and the 1D outputSample */
      NumericalScalar CorrelationAnalysis::SpearmanCorrelation(const NumericalSample & inputSample,
							       const NumericalSample & outputSample,
							       const UnsignedLong index)
      {
	if (index >= inputSample.getDimension()) throw InvalidArgumentException(HERE) << "Error: given index out of bound";
	if (outputSample.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: output sample must be 1D";
	if (inputSample.getSize() != outputSample.getSize()) throw InvalidArgumentException(HERE) << "Error: input and output samples must have the same size";
	return PearsonCorrelation(inputSample.rank(), outputSample.rank());
      }

      /* Compute the Standard Regression Coefficients (SRC) between the input sample and the output sample */
      CorrelationAnalysis::NumericalPoint CorrelationAnalysis::SRC(const NumericalSample & inputSample,
								   const NumericalSample & outputSample)
      {
	if (outputSample.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: output sample must be 1D";
	if (inputSample.getSize() != outputSample.getSize()) throw InvalidArgumentException(HERE) << "Error: input and output samples must have the same size";
	UnsignedLong dimension(inputSample.getDimension());
	LinearModel linear(LinearModelFactory().buildLM(inputSample, outputSample));
	NumericalPoint regression(linear.getRegression());
	NumericalScalar varOutput(outputSample.computeVariancePerComponent()[0]);
	NumericalPoint src(inputSample.computeVariancePerComponent());
	for (UnsignedLong i = 0; i < dimension; i++)
	  {
	    src[i] *= regression[i + 1] * regression[i + 1] / varOutput;
	  }
	return src;
      }

      /* Compute the Partial Correlation Coefficients (PCC) between the input sample and the output sample */
      CorrelationAnalysis::NumericalPoint CorrelationAnalysis::PCC(const NumericalSample & inputSample,
								   const NumericalSample & outputSample)
      {
	if (inputSample.getDimension() < 2) throw InvalidDimensionException(HERE) << "Error: input sample must have dimension > 1";
	if (outputSample.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: output sample must be 1D";
	if (inputSample.getSize() != outputSample.getSize()) throw InvalidArgumentException(HERE) << "Error: input and output samples must have the same size";
	UnsignedLong dimension(inputSample.getDimension());
	UnsignedLong size(inputSample.getSize());
	NumericalPoint pcc(dimension);
	// For each component i, perform an analysis on the truncated input sample where Xi has been removed
	NumericalSample truncatedInput(size, dimension - 1);
	NumericalSample remainingInput(size, 1);
	for (UnsignedLong index = 0; index < dimension; index++)
	  {
	    // Build the truncated sample
	    for (UnsignedLong i = 0; i < size; i++)
	      {
		for (UnsignedLong j = 0; j < index; j++)
		  {
		    truncatedInput[i][j] = inputSample[i][j];
		  }
		for (UnsignedLong j = index + 1; j < dimension; j++)
		  {
		    truncatedInput[i][j - 1] = inputSample[i][j];
		  }
		remainingInput[i][0] = inputSample[i][index];
	      }
	    // Build the linear models
	    LinearModel outputVersusTruncatedInput(LinearModelFactory().buildLM(truncatedInput, outputSample));
	    LinearModel remainingVersusTruncatedInput(LinearModelFactory().buildLM(truncatedInput, remainingInput));
	    // Compute the correlation between the residuals
	    NumericalSample residualOutput(outputVersusTruncatedInput.getResidual(truncatedInput, outputSample));
	    NumericalSample residualRemaining(remainingVersusTruncatedInput.getResidual(truncatedInput, remainingInput));
	    pcc[index] = PearsonCorrelation(residualOutput, residualRemaining);
	  }
	return pcc;
      }

      /* Compute the Standard Rank Regression Coefficients (SRRC) between the input sample and the output sample */
      CorrelationAnalysis::NumericalPoint CorrelationAnalysis::SRRC(const NumericalSample & inputSample,
								    const NumericalSample & outputSample)
      {
	if (outputSample.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: output sample must be 1D";
	if (inputSample.getSize() != outputSample.getSize()) throw InvalidArgumentException(HERE) << "Error: input and output samples must have the same size";
	return SRC(inputSample.rank(), outputSample.rank());
      }

      /* Compute the Partial Rank Correlation Coefficients (PRCC) between the input sample and the output sample */
      CorrelationAnalysis::NumericalPoint CorrelationAnalysis::PRCC(const NumericalSample & inputSample,
								    const NumericalSample & outputSample)
      {
	// Perform the basic checks of the inputs, to avoid costly ranking if finally PCC will fail
	if (inputSample.getDimension() < 2) throw InvalidDimensionException(HERE) << "Error: input sample must have dimension > 1";
	if (outputSample.getDimension() != 1) throw InvalidDimensionException(HERE) << "Error: output sample must be 1D";
	if (inputSample.getSize() != outputSample.getSize()) throw InvalidArgumentException(HERE) << "Error: input and output samples must have the same size";
	return PCC(inputSample.rank(), outputSample.rank());
      }

    } // namespace Stat
  } // namespace Base
} // namespace OpenTURNS
