//                                               -*- C++ -*-
/**
 *  @file  TNC.hxx
 *  @brief TNC is an actual implementation for a bound-constrained optimization algorithm
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: TNC.hxx 995 2008-10-31 10:52:04Z dutka $
 */
#ifndef OPENTURNS_TNC_HXX
#define OPENTURNS_TNC_HXX

#include "OT.hxx"
#include "BoundConstrainedAlgorithmImplementation.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Optimisation
    {

      /**
       * @class TNC
       * TNC is an actual implementation for
       * BoundConstrainedAlgorithm using the TNC library
       */
      class TNC :
	public BoundConstrainedAlgorithmImplementation
      {
	CLASSNAME;
      public:

	typedef BoundConstrainedAlgorithmImplementation::NumericalMathFunction NumericalMathFunction;
	typedef BoundConstrainedAlgorithmImplementation::InternalException     InternalException;

#ifndef SWIG
	/**
	 * @class SpecificParameters
	 * Specific parameters of the TNC library
	 */
	class SpecificParameters
	  : public Common::PersistentObject
	{
	  static const UnsignedLong    DefaultMaxCGit;
	  static const NumericalScalar DefaultEta;
	  static const NumericalScalar DefaultStepmx;
	  static const NumericalScalar DefaultAccuracy;
	  static const NumericalScalar DefaultFmin;
	  static const NumericalScalar DefaultRescale;

	  CLASSNAME;
	public:
	  /** Default constructor */
	  SpecificParameters();

	  /** Parameters constructor */
	  SpecificParameters(const NumericalPoint & scale,
			     const NumericalPoint & offset,
			     const UnsignedLong    maxCGit,
			     const NumericalScalar eta,
			     const NumericalScalar stepmx,
			     const NumericalScalar accuracy,
			     const NumericalScalar fmin,
			     const NumericalScalar rescale);

	  /** Virtual constructor */
	  virtual SpecificParameters * clone() const;

	  /** Scale accessor */
	  NumericalPoint getScale() const;
	  void setScale(const NumericalPoint & scale);

	  /** Offset accessor */
	  NumericalPoint getOffset() const;
	  void setOffset(const NumericalPoint & offset);

	  /** MaxCGit accessor */
	  UnsignedLong getMaxCGit() const;
	  void setMaxCGit(const UnsignedLong maxCGit);

	  /** Eta accessor */
	  NumericalScalar getEta() const;
	  void setEta(const NumericalScalar eta);

	  /** Stepmx accessor */
	  NumericalScalar getStepmx() const;
	  void setStepmx(const NumericalScalar stepmx);

	  /** Accuracy accessor */
	  NumericalScalar getAccuracy() const;
	  void setAccuracy(const NumericalScalar accuracy);

	  /** Fmin accessor */
	  NumericalScalar getFmin() const;
	  void setFmin(const NumericalScalar fmin);

	  /** Rescale accessor */
	  NumericalScalar getRescale() const;
	  void setRescale(const NumericalScalar rescale);

	  /** String converter */
	  String str() const;

	private:
	  NumericalPoint  scale_;
	  NumericalPoint  offset_;
	  UnsignedLong    maxCGit_;
	  NumericalScalar eta_;
	  NumericalScalar stepmx_;
	  NumericalScalar accuracy_;
	  NumericalScalar fmin_;
	  NumericalScalar rescale_;
	}; /* class SpecificParameters */
#else
	typedef OT::Base::Optimisation::TNCSpecificParameters SpecificParameters;
#endif

	/** Default constructor */
	TNC();

	/** Constructor with parameters: no constraint, starting from the origin */
	TNC(const NumericalMathFunction & objectiveFunction,
	    const Bool verbose = false);

	/** Constructor with parameters: bound constraints, starting from the given point */
	TNC(const SpecificParameters & parameters,
	    const NumericalMathFunction & objectiveFunction,
	    const Interval & boundConstraints,
	    const NumericalPoint & startingPoint,
	    const OptimizationProblem optimization  = MINIMIZATION,
	    const Bool verbose = false) throw(InvalidArgumentException);

	/** Virtual constructor */
	virtual TNC * clone() const;

	/** Performs the actual computation. Must be overloaded by the actual optimisation algorithm */
	void run()
	  throw(InternalException);

	/** Specific parameters accessor */
	SpecificParameters getSpecificParameters() const;

	/** Specific parameters accessor */
	void setSpecificParameters(const SpecificParameters & specificParameters);
	/** String converter */
	String str() const;

      private:

	/**
	 * Function that computes the objective function and its gradient
	 * corrected to correspond with the optimization problem (minimization or maximization)
	 */
	static int ObjectiveAndConstraint(double *x, double *f, double *g, void *state);
	SpecificParameters specificParameters_;
      }; /* class TNC */

      typedef OpenTURNS::Base::Optimisation::TNC::SpecificParameters  TNCSpecificParameters;

    } /* namespace Optimisation */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_TNC_HXX */
