//                                               -*- C++ -*-
/**
 *  @file  AbdoRackwitz.cxx
 *  @brief AbdoRackwitz is an actual implementation for
 *         NearestPointAlgorithm using the AbdoRackwitz algorithm.
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: AbdoRackwitz.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include <cmath>
#include "AbdoRackwitz.hxx"
#include "Log.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Optimisation
    {

      typedef Common::Log Log;

      CLASSNAMEINIT(AbdoRackwitz);

      /* Default constructor */
      AbdoRackwitz::AbdoRackwitz():
        NearestPointAlgorithmImplementation()
      {
	// Nothing to do
      }

      /*
       * @brief  Standard constructor: the problem is defined by a scalar valued function  (in fact, a 1-D vector valued fonction)
       *         and a level value
       */
      AbdoRackwitz::AbdoRackwitz(const SpecificParameters & specificParameters,
				 const NumericalMathFunction & levelFunction):
	NearestPointAlgorithmImplementation(levelFunction),
	specificParameters_(specificParameters),
	currentSigma_(0.0),
	currentPoint_(getStartingPoint().getDimension()),
	currentDirection_(getStartingPoint().getDimension()),
	currentLevelValue_(0.0),
	currentGradient_(getStartingPoint().getDimension()),
	currentLambda_(0.0)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      AbdoRackwitz * AbdoRackwitz::clone() const
      {
	return new AbdoRackwitz(*this);
      }

      /* Line search for globalization of the algorithm */
      NumericalScalar AbdoRackwitz::computeLineSearch()
      {
	/* Local copy of line search parameters */
	const NumericalScalar tau(specificParameters_.getTau());
	const NumericalScalar omega(specificParameters_.getOmega());
	const NumericalScalar smooth(specificParameters_.getSmooth());
	/* Logal copy of the level function and the level value */
	const NumericalMathFunction levelFunction(getLevelFunction());
	const NumericalScalar levelValue(getLevelValue());
	/* Actualize sigma */
	currentSigma_ = std::max(currentSigma_ + 1.0, smooth * currentPoint_.norm() / currentGradient_.norm());
	/* Compute penalized scalar objective function at current point */
	const NumericalScalar currentTheta(0.5 * currentPoint_.norm2() + currentSigma_ * fabs(currentLevelValue_ - levelValue));
	/* Min bound for step */
	const NumericalScalar minStep(getMaximumAbsoluteError() / currentDirection_.norm());
	/* Minimum decrease for the penalized objective function */
	const NumericalScalar levelIncrement(omega * NumericalPoint::dot(currentPoint_ + (currentSigma_ * ((currentLevelValue_ > levelValue) ? 1.0 : -1.0)) * currentGradient_, currentDirection_));
	/* Initialization of the line search */
	/* We start with step=1 */
	NumericalScalar step(1.0);
	NumericalPoint currentStepPoint(NumericalPoint(currentPoint_.getDimension()));
	NumericalScalar currentStepLevelValue;
	NumericalScalar currentStepTheta;
	do
	  {
	    currentStepPoint = currentPoint_ + step * currentDirection_;
	    currentStepLevelValue = levelFunction(currentStepPoint)[0];
	    currentStepTheta = 0.5 * currentStepPoint.norm2() + currentSigma_ * fabs(currentStepLevelValue - levelValue);
	    if (getVerbose()) Log::Info(OSS() << "line search step=" << step << " currentStepPoint=" << currentStepPoint << " currentStepLevelValue=" << currentStepLevelValue << " currentStepTheta=" << currentStepTheta);
	    step *= tau;
	  }
	while ((step >= minStep) && ( currentStepTheta > currentTheta + step * levelIncrement));
	currentPoint_ = currentStepPoint;
	currentLevelValue_ = currentStepLevelValue;
	/* We went one step beyond */
	return step / tau;
      }

      /* Performs the actual computation by using the AbdoRackwitz algorithm
       */
      void AbdoRackwitz::run()
	throw(InternalException)
      {
	/* Get a local copy of the level function */
	const NumericalMathFunction levelFunction(getLevelFunction());
	/* Get a local copy of the level value */
	const NumericalScalar levelValue(getLevelValue());
	/* Current point -> u */
	currentPoint_ = getStartingPoint();
	Bool convergence(false);
	UnsignedLong iterationCount(1);
	NumericalScalar absoluteError(-1.0);
	NumericalScalar constraintError(-1.0);
	NumericalScalar relativeError(-1.0);
	NumericalScalar residualError(-1.0);
	/* Compute the level function at the current point -> G */
	currentLevelValue_ = levelFunction(currentPoint_)[0];
	while ( (!convergence) && (iterationCount <= getMaximumIterationsNumber()) )
	  {
	    /* Compute the level function gradient at the current point -> Grad(G) */
	    currentGradient_ = levelFunction.gradient(currentPoint_) * NumericalPoint(1, 1.0);
	    if (getVerbose()) Log::Info(OSS() << "current point=" << currentPoint_ << " current level value=" << currentLevelValue_ << " current gradient=" << currentGradient_);
	    /* Compute the current Lagrange multiplier */
	    const NumericalScalar normGradientSquared(currentGradient_.norm2());
	    /* In case of a null gradient, throw an internal exception */
	    if (normGradientSquared == 0)
	      {
		setResult(Result(currentPoint_, iterationCount, absoluteError, relativeError, residualError, constraintError));
		throw InternalException(HERE) << "Error in Abdo Rackwitz algorithm: the gradient of the level function is zero at point u=" << currentPoint_;
	      }
	    /* Lambda = (G - <Grad(G), u>) / ||Grad(G)||^2 */
	    currentLambda_ = (currentLevelValue_ - levelValue - NumericalPoint::dot(currentGradient_, currentPoint_)) / normGradientSquared;
	    /* Compute the current direction Du = -Lambda Grad(G) - u */
	    /* Be careful! currentGradient_ is an n by 1 matrix, we must multiply it by a 1 by 1
	     * vector in order to get an n dimensional equivalente vector
	     */
	    currentDirection_ = -currentLambda_ * currentGradient_ - currentPoint_;
	    /* Perform a line search in the given direction */
	    const NumericalScalar alpha(computeLineSearch());
	    /* Check if convergence has been achieved */
	    absoluteError = fabs(alpha) * currentDirection_.norm();
	    constraintError = fabs(currentLevelValue_ - levelValue);
	    const NumericalScalar pointNorm(currentPoint_.norm());
	    if (pointNorm > 0.0)
	      {
		relativeError = absoluteError / pointNorm;
	      }
	    else
	      {
		relativeError = -1.0;
	      }
	    residualError = (currentPoint_ + currentLambda_ * currentGradient_).norm();
	    convergence = ((absoluteError < getMaximumAbsoluteError()) && (relativeError < getMaximumRelativeError())) || ((residualError < getMaximumResidualError()) && (constraintError < getMaximumConstraintError()));
	    setResult(Result(currentPoint_, iterationCount, absoluteError, relativeError, residualError, constraintError));
	    Log::Info(getResult().str());
	    /* Go to next iteration */
	    iterationCount++;
	  }
	setResult(Result(currentPoint_, iterationCount, absoluteError, relativeError, residualError, constraintError));
	/* Check if we converged */
	if (!convergence)
	  {
	    throw InternalException(HERE) << "Error in Abdo Rackwitz algorithm: no convergence after " << iterationCount << " iterations";
	  }
      } // run()

      /* Specific parameters accessor */
      AbdoRackwitz::SpecificParameters AbdoRackwitz::getSpecificParameters() const
      {
	return specificParameters_;
      }

      /* Specific parameters accessor */
      void AbdoRackwitz::setSpecificParameters(const SpecificParameters & specificParameters)
      {
	specificParameters_ = specificParameters;
      }

      /* String converter */
      String AbdoRackwitz::str() const
      {
	OSS oss;
	oss << "class=" << AbdoRackwitz::GetClassName()
	    << " " << NearestPointAlgorithmImplementation::str()
	    << " specificParameters=" << getSpecificParameters();
	return oss;
      }

    } /* namespace Optimisation */
  } /* namespace Base */
} /* namespace OpenTURNS */
