//                                               -*- C++ -*-
/**
 *  @file  Staircase.cxx
 *  @brief Staircase class for staircase plots
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (mer 21 mai 2008) $
 *  Id:      $Id: Staircase.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "Staircase.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Graph {

      CLASSNAMEINIT(Staircase);
      
      static Common::Factory<Staircase> RegisteredFactory("Staircase");

      /* Default constructor */
      Staircase::Staircase(const NumericalSample & data,
			   const String & legend):
	DrawableImplementation(data, legend),
	pattern_(DefaultPattern)
      {
	// Check data validity
	setData(data);
      }

      /* Constructor with parameters */
      Staircase::Staircase(const NumericalSample & data,
			   const String & color,
			   const String & lineStyle,
			   const String & pattern,
			   const String & legend)
	throw(InvalidArgumentException):
	DrawableImplementation(data, legend)
      {
	setColor(color);
	setLineStyle(lineStyle);
	setPattern(pattern);
	// Check data validity
	setData(data);
      }
      
      /* String converter */
      String Staircase::str() const
      {
	OSS oss;
	oss << "class=" << Staircase::GetClassName()
	    << " name=" << getName()
	    << " pattern=" << pattern_
	    << " derived from " << DrawableImplementation::str();
	return oss;
      }

      /* Draw method */
      String Staircase::draw() const
      {
	OSS oss;
	// Stores the data in a temporary file
	oss << DrawableImplementation::draw() << "\n";
	// The specific R command for drawing
	oss << "lines(data[,1], data[,2]"
	    << ",lty=\"" << lineStyle_
	    << "\",col=\"" << color_
	    << "\",type=\"" << pattern_
	    << "\")";
	
	return oss;
      }
      
      /* Clone method */
      Staircase * Staircase::clone() const
      {
	return new Staircase(*this);
      }

      /* Accessor for pattern */
      String Staircase::getPattern() const
      {
	return pattern_;
      }

      /* Accessor for pattern */
      void Staircase::setPattern(const String style) throw(InvalidArgumentException)
      {
	if(!isValidPattern(style)) throw InvalidArgumentException(HERE) << "Given staircase pattern = " << style << " is incorrect";
	pattern_ = style;
      }

      /* Check validity of data */
      Bool Staircase::isValidData(const NumericalSample & data) const
      {
	return (data.getDimension() == 2);
      }

      /* Method save() stores the object through the StorageManager */
      void Staircase::save(const StorageManager::Advocate & adv) const
      {
      	DrawableImplementation::save(adv);
      	adv.writeValue("pattern_", pattern_);
      }

      /* Method load() reloads the object from the StorageManager */
      void Staircase::load(const StorageManager::Advocate & adv)
      {
      	DrawableImplementation::load(adv);
	String name;
	String value;
	StorageManager::List objList = adv.getList(StorageManager::StringEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, value)) {
	    if (name == "pattern_") pattern_ = value;
	  }
	}
      }

    } /* namespace Graph */

  }/* namespace Base */

}/* namespace OpenTURNS */
