//                                               -*- C++ -*-
/**
 *  @file  Pie.cxx
 *  @brief Pie class for piechart plots
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-05-21 17:44:02 +0200 (mer 21 mai 2008) $
 *  Id:      $Id: Pie.cxx 818 2008-05-21 15:44:02Z dutka $
 */
#include "Pie.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Graph {

      CLASSNAMEINIT(Pie);  
    
      static Common::Factory<Pie> RegisteredFactory("Pie");

      const NumericalScalar Pie::HorizontalMargin = 0.3;
      const NumericalScalar Pie::VerticalMargin = 0.1;
      const NumericalScalar Pie::LabelThreshold = 0.02;

      /* Default constructor */
      Pie::Pie(const NumericalSample & data) :
	DrawableImplementation(data, NoSpecifiedLabel),
	palette_(data.getSize()),
	radius_(1),
	center_(2),
	labels_(data.getSize())
      {
	for (UnsignedLong i = 0; i < data.getSize(); ++i)
	  {
	    palette_[i] = DefaultSurfaceColor;
	  }
	// Check data validity
	setData(data);
      }
      
      /* Constructor with parameters */
      Pie::Pie(const NumericalSample & data,
	       const Description & labels,
	       const NumericalPoint & center,
	       const NumericalScalar & radius,
	       const Description & palette)
	throw(InvalidArgumentException) : 
	DrawableImplementation(data, NoSpecifiedLabel),
	palette_(),
	radius_(radius),
	center_(center),
	labels_(labels)
      {
	if(!isValidColorPalette(palette)) throw InvalidArgumentException(HERE) << "Given color palette = " << palette << " is incorrect";
	// Check data validity
	setData(data);
	palette_ = palette;
      }
      
      /* String converter */
      String Pie::str() const
      {
	OSS oss;
	oss << "class=" << Pie::GetClassName()
	    << " name=" << getName()
	    << " labels=" << labels_
	    << " radius=" << radius_
	    << " center=" << center_
	    << " color palette=" << palette_
	    << " derived from " << DrawableImplementation::str();
	return oss;
      }
      
      /* Accessor for center */
      Pie::NumericalPoint Pie::getCenter() const
      {
	return center_;
      }
      void Pie::setCenter(const NumericalPoint & center)
      {
	center_ = center;
      }

      /* Accessor for radius */
      NumericalScalar Pie::getRadius() const
      {
	return radius_;
      }
      void Pie::setRadius(const NumericalScalar radius)
      {
	radius_ = radius;
      }
	
      /* Accessor for labels */
      Pie::Description Pie::getLabels() const
      {
	return labels_;
      }
      void Pie::setLabels(const Description & labels)
      {
	labels_ = labels;
      }

      /* Accessor for color palette */
      Pie::Description Pie::getPalette() const
      {
	return palette_;
      }
      void Pie::setPalette(const Description & palette)
	throw(InvalidArgumentException)
      {
	if(!isValidColorPalette(palette)) throw InvalidArgumentException(HERE) << "Given color palette = " << palette << " is incorrect";
	palette_ = palette;
      }

      /* Accessor for boundingbox */
      Pie::BoundingBox Pie::getBoundingBox() const
      {
	BoundingBox boundingBox(BoundingBoxSize);
	boundingBox[0] = center_[0] - (1.0 + HorizontalMargin) * radius_;
	boundingBox[1] = center_[0] + (1.0 + HorizontalMargin) * radius_;
	boundingBox[2] = center_[1] - (1.0 + VerticalMargin) * radius_;
	boundingBox[3] = center_[1] + (1.0 + VerticalMargin) * radius_;
	return boundingBox;
      }

      /* Draw method */
      String Pie::draw() const
      {
	//pie(xpi,center=center,radius=rayon,labels=noms)
	OSS oss;
	// Stores the data in a temporary file
	oss << DrawableImplementation::draw() << "\n";
	// The specific R command for drawing
	// Labels are drawn only if the associated data shares a sufficient amount of the total
	NumericalScalar labelThreshold(data_.getMax()[0] * LabelThreshold);
	oss << "pie(data[,1],"
            << "center=c(" << center_[0] << "," << center_[1]
            << "),radius=" << radius_;
        UnsignedLong size(labels_.getSize());
	// If there is any label defined
	if (size > 0)
	  {
            oss << ",labels=c(\"";
	    String separator("");
	    for(UnsignedLong i = 0; i < size; ++i, separator = "\",\"")
	      {
		oss << separator << (data_[i][0] >= labelThreshold ? labels_[i] : "");
	      }
	    oss << "\")";
	  }
	size = palette_.getSize();
	// If there is any color defined
	if (size > 0)
	  {
	    oss << ",col=c(\"";
	    String separator("");
	    for(UnsignedLong i = 0; i < size; ++i, separator = "\",\"")
	      {
		oss << separator << palette_[i];
	      }
	    oss << "\")";
	  }
	oss << ")";
	return oss;
      }

      /* Clone method */
      Pie * Pie::clone() const
      {
	return new Pie(*this);
      }

      /* Check for color palette validity */
      Bool Pie::isValidColorPalette(const Description & palette)
      {
	Bool isValid = true;
	Description::const_iterator it;
	for(it = palette.begin(); it != palette.end(); it++)
	  isValid &= isValidColor(*it);
	return isValid;
      }
      
      /* Check for data validity */
      Bool Pie::isValidData(const NumericalSample & data) const
      {
	// First, check the dimension
	if (data.getDimension() != 1) return false;
	UnsignedLong size(data.getSize());
	// Check if there is any data to display
	if (size == 0) return false;
	// Then, check the positivity of the data
	NumericalScalar max(0.0);
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    NumericalScalar x(data[i][0]);
	    if (x < 0.0) return false;
	    if (x > max) max = x;
	  }
	// Check if there is at least one strictly positive data
	if (max == 0.0) return false;
	// Everything is ok
	return true;
      }
      
      /* Method save() stores the object through the StorageManager */
      void Pie::save(const StorageManager::Advocate & adv) const
      {
      	DrawableImplementation::save(adv);
	adv.writeValue(palette_, StorageManager::MemberNameAttribute, "palette_");
	adv.writeValue("radius_", radius_);
	adv.writeValue(center_, StorageManager::MemberNameAttribute, "center_");
	adv.writeValue(labels_, StorageManager::MemberNameAttribute, "labels_");
      }

      /* Method load() reloads the object from the StorageManager */
      void Pie::load(const StorageManager::Advocate & adv)
      {
      	DrawableImplementation::load(adv);
	adv.readValue(palette_, StorageManager::MemberNameAttribute, "palette_");
	String name;
	NumericalScalar numericalScalarValue;
	StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, numericalScalarValue)) {
	    if (name == "radius_") radius_ = numericalScalarValue;
	  }
	}
	adv.readValue(center_, StorageManager::MemberNameAttribute, "center_");
	adv.readValue(labels_, StorageManager::MemberNameAttribute, "labels_");
      }

    }/* namespace Graph */
    
  }/*namespace Base */
  
}/* namespace OpenTURNS */
