//                                               -*- C++ -*-
/**
 *  @file  BarPlot.cxx
 *  @brief BarPlot class for barplots
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-09-19 11:50:28 +0200 (ven 19 sep 2008) $
 *  Id:      $Id: BarPlot.cxx 937 2008-09-19 09:50:28Z dutka $
 */
#include "BarPlot.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Graph {

      CLASSNAMEINIT(BarPlot);
     
      static Common::Factory<BarPlot> RegisteredFactory("BarPlot");

      /** Default constructor */
      BarPlot::BarPlot(const NumericalSample & data,
		       const NumericalScalar origin,
		       const String & legend):
	DrawableImplementation(data, legend),
	origin_(origin)
      {
	// Check data validity
	setData(data);
      }
      
      /** Constructor with parameters */
      BarPlot::BarPlot(const NumericalSample & data,
		       const NumericalScalar origin,
		       const String & color,
		       const String & fillStyle,
		       const String & lineStyle,
		       const String & legend)
	throw(InvalidArgumentException):
	DrawableImplementation(data, legend),
	origin_(origin)
      {
	// Check data validity
	setData(data);
	setColor(color);
	setFillStyle(fillStyle);
	setLineStyle(lineStyle);
      }

      /** String converter */
      String BarPlot::str() const
      {
	OSS oss;
	oss << "class=" << BarPlot::GetClassName()
	    << " name=" << getName()
	    << " origin=" << origin_
	    << " derived from " << DrawableImplementation::str();
	return oss;
      }

      /** Accessor for origin */
      NumericalScalar BarPlot::getOrigin() const
      {
	return origin_;
      }
      
      /** Accessor for origin */
      void BarPlot::setOrigin(const NumericalScalar origin)
      {
	origin_ = origin;
      }

      /** Accessor for boundingbox */
      BarPlot::BoundingBox BarPlot::getBoundingBox() const
      {
	BoundingBox boundingBox(BoundingBoxSize);
	const Type::NumericalPoint max(data_.getMax());
	const Type::NumericalPoint min(data_.getMin());
	boundingBox[0] = origin_;
	boundingBox[1] = origin_;
	const UnsignedLong size(data_.getSize());
	for(UnsignedLong i = 0; i < size; ++i)
	  {
	    boundingBox[1] += data_[i][0];
	  }
	boundingBox[2] = std::min(0.0, min[1]);
	boundingBox[3] = std::max(0.0, max[1]);
	return boundingBox;
      }

      /** Draw method */
      String BarPlot::draw() const
      {
	OSS oss;
	// Stores the data in a temporary file
	oss << DrawableImplementation::draw() << "\n";
	// The specific R command for drawing
	oss << "x<-data[,1];"
	    << "y<-data[,2];"
	    << "n<-length(y)" << ";"
	    << "x<-cumsum(c(" << origin_ << ", x));"
	    << "rect(x[1:n],rep(0,n),x[2:(n+1)],y,col=\"" << color_
	    << "\",lty=\"" << lineStyle_;

	if(getFillStyle() == "shaded")
	  oss << "\",density=20)";
	else oss << "\")";

	return oss;
      }
      
      /** Clone method */
      BarPlot * BarPlot::clone() const
      {
	return new BarPlot(*this);
      }

      /** Check validity of data */
      Bool BarPlot::isValidData(const NumericalSample & data) const
      {
	return (data.getDimension() == 2);
      }

      /* Method save() stores the object through the StorageManager */
      void BarPlot::save(const StorageManager::Advocate & adv) const
      {
      	DrawableImplementation::save(adv);
      	adv.writeValue("origin_", origin_);
      }

      /* Method load() reloads the object from the StorageManager */
      void BarPlot::load(const StorageManager::Advocate & adv)
      {
	DrawableImplementation::load(adv);

	String name;
	NumericalScalar value;
	StorageManager::List objList = adv.getList(StorageManager::NumericalScalarEntity);
	for(objList.firstValueToRead(); objList.moreValuesToRead(); objList.nextValueToRead()) {
	  if (objList.readValue(name, value)) {
	    if (name == "origin_") origin_ = value;
	  }
	}
      }

    } /* namespace Graph */

  }/* namespace Base */

}/* namespace OpenTURNS */
