//                                               -*- C++ -*-
/**
 *  @file  WrapperObject.hxx
 *  @brief An object that binds itself to an external shared library
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-17 14:26:02 +0200 (ven 17 oct 2008) $
 *  Id:      $Id: WrapperObject.hxx 974 2008-10-17 12:26:02Z dutka $
 */
#ifndef OPENTURNS_WRAPPEROBJECT_HXX
#define OPENTURNS_WRAPPEROBJECT_HXX

#include "PersistentObject.hxx"
#include "Library.hxx"
#include "WrapperData.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"
#include "Matrix.hxx"
#include "SymmetricTensor.hxx"
#include "WrapperInterface.h"
#include "Exception.hxx"
#include "Description.hxx"
#include "Pointer.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Func
    {

      /**
       * @class WrapperObject
       *
       * A WrapperObject offers an interface for any object (function,
       * gradient, hessian) that is located in a external shared
       * library.
       */

      class WrapperObject
	: public Common::PersistentObject
      {
	CLASSNAME;

      public:

	typedef Common::WrapperInternalException WrapperInternalException;
	typedef Common::InvalidArgumentException InvalidArgumentException;
	typedef Type::NumericalPoint             NumericalPoint;
	typedef Stat::NumericalSample            NumericalSample;
	typedef Type::Matrix                     Matrix;
	typedef Type::SymmetricTensor            SymmetricTensor;
	typedef Type::Description                Description;

	enum ObjectType {
	  FUNCTION = 0,
	  GRADIENT,
	  HESSIAN,
	  ObjectPrefixSize };

	enum FunctionType {
	  STATE_CREATION = 0,
	  STATE_DELETION,
	  INFORMATION,
	  INITIALIZATION,
	  EXECUTION,
	  EXECUTION_SAMPLE,
	  FINALIZATION,
	  FunctionPrefixSize };

      private:

	class ObjectPrefix : public std::vector<String>
	{
	public:
	  ObjectPrefix();
	}; /* class ObjectPrefix */

	/** Array of prefix according to object type */
	static const ObjectPrefix ObjectPrefix_;

	class FunctionPrefix : public std::vector<String>
	{
	public:
	  FunctionPrefix();
	}; /* class FunctionPrefix */

	/** Array of prefix according to function type */
	static const FunctionPrefix FunctionPrefix_;




      public:

	/** 
	 * Constructor
	 *
	 * @param libraryPath the path of the library to load
	 * @param symbolName  the name of the symbol to bind to
	 * @param data        the data to send to the wrapper
	 * @param o           the kind of this object
	 */
	WrapperObject(const FileName & libraryPath,
		      const String & symbolName,
		      const WrapperData & data,
		      ObjectType o)
	  throw (WrapperInternalException);

	/** Virtual constructor */
	virtual WrapperObject * clone() const;

	/** Comparison operator */
	Bool operator ==(const WrapperObject & other) const;

	/**
	 * String converter
	 * This method shows human readable information on the
	 * internal state of an object. It is used when streaming
	 * the object or for user information.
	 */
	virtual String str() const;

	/**
	 * Method getFunctionName returns the name of the symbol
	 * in the shared library.
	 *
	 * @param f the kind of function we want to bind to
	 */
	String getFunctionName(FunctionType f) const;

	/** Data accessor */
	void setWrapperData(const WrapperData & data);
	const WrapperData & getWrapperData() const;


	/** Method getInNumericalPointDimension returns the dimension of the in point */
	UnsignedLong getInNumericalPointDimension(void * p_state) const
	  throw (WrapperInternalException);

	/** Method getInNumericalPointDimension returns the dimension of the out point */
	UnsignedLong getOutNumericalPointDimension(void * p_state) const
	  throw (WrapperInternalException);

	/** Method getDescription returns the description of the input and output
	 * variables of the functions
	 */
	Description getDescription() const
	  throw (WrapperInternalException);

	/** Method initialize calls the initializationSymbol of the library */
	void initialize(void * p_state) const
	  throw (WrapperInternalException);

	/** Method execute actually realizes the desired treatment */
 	NumericalPoint execute(void * p_state, const NumericalPoint & in) const 
	  throw(InvalidArgumentException,WrapperInternalException);

	/** Method execute actually realizes the desired treatment */
 	NumericalSample execute(void * p_state, const NumericalSample & in) const 
	  throw(InvalidArgumentException,WrapperInternalException);

	/** Method gradient actually realizes the desired treatment */
 	Matrix gradient(void * p_state, const NumericalPoint & in) const 
	  throw(InvalidArgumentException,WrapperInternalException);

	/** Method hessian actually realizes the desired treatment */
 	SymmetricTensor hessian(void * p_state, const NumericalPoint & in) const 
	  throw(InvalidArgumentException,WrapperInternalException);

	/** Method finalize calls the finalizationSymbol of the library */
	void finalize(void * p_state) const
	  throw (WrapperInternalException);

	/** State management */
	void * createNewState()	throw (WrapperInternalException);
	void   deleteState(void * p_state) throw (WrapperInternalException);


      protected:

	/** Symbol accessor */
	LibrarySymbol getSymbol(const String & name, Bool optional = false, LibrarySymbol alternateFunction = 0) const
	  throw (WrapperInternalException);

      private:

	/** The library handle */
	Library handle_;

	/** The data send to the wrapper */
	WrapperData data_;

	/** The kind of object we are */
	ObjectType type_;

	/** The library symbols */
	GetWrapperInformationFunctionPointer getInfoSymbol_;
	InitializationFunctionPointer initSymbol_;
	ExecutionFunctionPointer execSymbol_;
	ExecutionSampleFunctionPointer execSampleSymbol_;
	GradientFunctionPointer gradSymbol_;
	HessianFunctionPointer hessSymbol_;
	FinalizationFunctionPointer finalizeSymbol_;
	StateCreationFunctionPointer stateCreationSymbol_;
	StateDeletionFunctionPointer stateDeletionSymbol_;

	/* The information structure */
	mutable Common::Pointer<struct WrapperInformation> wrapperInfo_;

      }; /* class WrapperObject */



    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_WRAPPEROBJECT_HXX */
