//                                               -*- C++ -*-
/**
 *  @file  WrapperFile.cxx
 *  @brief This class provides all the treatments for wrapper file manipulation
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-29 18:54:45 +0100 (mer 29 oct 2008) $
 *  Id:      $Id: WrapperFile.cxx 990 2008-10-29 17:54:45Z dutka $
 */
#include <vector>                 // for std::vector
#include <string>                 // for std::string
#include <cstdlib>                // for getenv
#include <sys/types.h>            // for stat
#include <sys/stat.h>             // for stat
#include <unistd.h>               // for stat
#include "OSS.hxx"
#include "WrapperFile.hxx"
#include "Path.hxx"
#include "Exception.hxx"
#include "Log.hxx"
#include "WrapperCommon.h"
#include "XMLToolbox.hxx"

#ifdef XML_SUPPORTED
#include "XMLTags.hxx"
#endif

namespace OpenTURNS
{

  namespace Base
  {

    namespace Func
    {

      using Common::XMLParserException;
      using Common::WrapperFileParsingException;
      using Common::FileNotFoundException;
      using Common::Log;
      using Common::Path;

      /* The file name extension */
#ifdef XML_SUPPORTED
      const String WrapperFile::extension_ = ".xml";
#else
      const String WrapperFile::extension_ = ".txt";
#endif

      /* The environment variable name */
      const String WrapperFile::openturnsWrapperPathVariableName_ = "OPENTURNS_WRAPPER_PATH";

      /* The HOME subdirectory path */
      const String WrapperFile::homeSubdirectory_ = "/openturns/wrappers";

      /* The 'prefix' subdirectory path */
      const String WrapperFile::prefixSubdirectory_ = "/openturns/wrappers";



      CLASSNAMEINIT(WrapperFile);

      /*
       * Default constructor
       */
      WrapperFile::WrapperFile(const FileName & pathToFile)
	throw(WrapperFileParsingException)
	: PersistentObject(),
	  descriptionFilePath_(pathToFile),
	  data_()
      {
	init();
	parseFile(pathToFile);
	done();
      }

      /* Virtual constructor */
      WrapperFile * WrapperFile::clone() const
      {
	return new WrapperFile(*this);
      }


      /* String converter */
      String WrapperFile::str() const
      {
	OSS oss;
	oss << "class=" << getClassName()
	    << " name=" << getName()
            << " data={" << data_
	    << "}";
	return oss;
      }

#if defined HAS_LIBXML2

      using Common::XMLDoc;
      using Common::XML;

      /* Standard Initialization */
      void WrapperFile::init() const
	throw(WrapperFileParsingException)
      {
	// Nothing to do
      }

      /* Standard finalization */
      void WrapperFile::done() const
      {
	// Nothing to do
      }


      /* Standard parsing function */
      void WrapperFile::parseFile(const FileName & pathToFile)
	throw(WrapperFileParsingException)
      {
	// Load the wrapper file
	XMLDoc doc( pathToFile );

	// Check it is an OpenTURNS' one
	xmlNodePtr wrapperElt = xmlDocGetRootElement( doc );
	if (wrapperElt == NULL) throw WrapperFileParsingException(HERE) << "Wrapper file has no root element (" << pathToFile << ")";
	if (xmlStrcmp( wrapperElt->name, REINTERPRET_CAST(const xmlChar *, XMLTag_wrapper) ))
	  throw WrapperFileParsingException(HERE) << "Wrapper file " << pathToFile
						  << " has an invalid root element (" << wrapperElt->name << ")"
						  << " at line " << xmlGetLineNo(wrapperElt);

	xmlNodePtr libraryElt     = XML::FindElementByName( wrapperElt, XMLTag_library );
	xmlNodePtr libraryPathElt = XML::FindElementByName( libraryElt, XMLTag_path );
	String  libraryPath       = XML::GetNodeValue( libraryPathElt );
	if (libraryPath.empty()) throw WrapperFileParsingException(HERE) << "Library path tag not found in wrapper file " << pathToFile;
	data_.setLibraryPath( libraryPath );

	Log::Debug(OSS() << "Read library  path                    : " << libraryPath );



	xmlNodePtr descriptionElt  = XML::FindElementByName( libraryElt, XMLTag_description );
	xmlNodePtr variableListElt = XML::FindElementByName( descriptionElt, XMLTag_variable_list );
	WrapperData::VariableListType variableList;
	if (variableListElt && variableListElt->children) {
	  for (xmlNodePtr current = variableListElt->children; current; current = current->next) {
	    if (XML::IsElement(current, XMLTag_variable)) {
	      WrapperDataVariable variable;

	      xmlNodePtr commentElt = XML::FindElementByName( current, XMLTag_comment );
	      if (commentElt) variable.comment_ = XML::GetNodeValue( commentElt );
	      
	      xmlNodePtr unitElt    = XML::FindElementByName( current, XMLTag_unit );
	      if (unitElt) variable.unit_ = XML::GetNodeValue( unitElt );
	      
	      xmlNodePtr regexpElt  = XML::FindElementByName( current, XMLTag_regexp );
	      if (regexpElt) variable.regexp_ = XML::GetNodeValue( regexpElt );
	      
	      xmlNodePtr formatElt  = XML::FindElementByName( current, XMLTag_format );
	      if (formatElt) variable.format_ = XML::GetNodeValue( formatElt );

	      variable.id_  = XML::GetAttributeByName( current, XMLTag_id );
	      
	      String type_ = XML::GetAttributeByName( current, XMLTag_type );
	      if (type_ == XMLAttr_in) variable.type_ = WrapperDataVariableType::IN;
	      else if (type_ == XMLAttr_out) variable.type_ = WrapperDataVariableType::OUT;
	      else {
		/* We should never go here if the wrapper file has been validated according to the DTD */
		throw WrapperFileParsingException(HERE) << "Unknown type (" << type_
							<< ") for variable in wrapper file " << pathToFile
							<< " at line " << xmlGetLineNo(current);
	      }
	      
	      String computedGradient_ = XML::GetAttributeByName( current, XMLTag_computed_gradient );
	      if (computedGradient_ == XMLAttr_yes) variable.gradient_ = WrapperComputedGradient::YES;
	      else if (computedGradient_ == XMLAttr_no) variable.gradient_ = WrapperComputedGradient::NO;
	      else if (computedGradient_.empty()) variable.gradient_ = WrapperComputedGradient::NO;
	      else {
		/* We should never go here if the wrapper file has been validated according to the DTD */
		throw WrapperFileParsingException(HERE) << "Unknown " << XMLTag_computed_gradient << " attribute (" << computedGradient_
							<< ") for variable in wrapper file " << pathToFile
							<< " at line " << xmlGetLineNo(current);
	      }
	      
	      Log::Debug(OSS() << "Read data variable id                 : " << variable.id_);
	      Log::Debug(OSS() << "Read data variable comment            : " << variable.comment_);
	      Log::Debug(OSS() << "Read data variable unit               : " << variable.unit_);
	      Log::Debug(OSS() << "Read data variable regexp             : " << variable.regexp_);
	      Log::Debug(OSS() << "Read data variable format             : " << variable.format_);
	      Log::Debug(OSS() << "Read data variable type               : " << WrapperListElementTypeAsString[variable.type_]);
	      Log::Debug(OSS() << "Read data variable computed-gradient  : " << WrapperProvidedAsString[variable.gradient_]);
	      
	      variableList.add( variable );
	    }
	  }
	}
	data_.setVariableList( variableList );



	WrapperFunctionDescription functionDesc;
	xmlNodePtr functionElt = XML::FindElementByName( descriptionElt, XMLTag_function );
	functionDesc.name_     = XML::GetNodeValue( functionElt );

	String functionProvided_  = XML::GetAttributeByName( functionElt, XMLTag_provided );
	if (functionProvided_ == XMLAttr_yes) functionDesc.provided_ = WrapperSymbolProvided::YES;
	else if (functionProvided_ == XMLAttr_no) functionDesc.provided_ = WrapperSymbolProvided::NO;
	else if (functionProvided_.empty()) functionDesc.provided_ = WrapperSymbolProvided::NO;
	else {
	  /* We should never go here if the wrapper file has been validated according to the DTD */
	  throw WrapperFileParsingException(HERE) << "Unknown " << XMLTag_provided << " attribute (" << functionProvided_
						  << ") for variable in wrapper file " << pathToFile
						  << " at line " << xmlGetLineNo(functionElt);
	}
	data_.setFunctionDescription( functionDesc );

	Log::Debug(OSS() << "Read function name                    : " << functionDesc.name_);
	Log::Debug(OSS() << "Read function name provided           : " << WrapperProvidedAsString[functionDesc.provided_]);




	WrapperFunctionDescription gradientDesc;
	xmlNodePtr gradientElt = XML::FindElementByName( descriptionElt, XMLTag_gradient );
	gradientDesc.name_     = XML::GetNodeValue( gradientElt );

	String gradientProvided_  = XML::GetAttributeByName( gradientElt, XMLTag_provided );
	if (gradientProvided_ == XMLAttr_yes) gradientDesc.provided_ = WrapperSymbolProvided::YES;
	else if (gradientProvided_ == XMLAttr_no) gradientDesc.provided_ = WrapperSymbolProvided::NO;
	else if (gradientProvided_.empty()) gradientDesc.provided_ = WrapperSymbolProvided::NO;
	else {
	  /* We should never go here if the wrapper file has been validated according to the DTD */
	  throw WrapperFileParsingException(HERE) << "Unknown " << XMLTag_provided << " attribute (" << gradientProvided_
						  << ") for variable in wrapper file " << pathToFile
						  << " at line " << xmlGetLineNo(gradientElt);
	}
	data_.setGradientDescription( gradientDesc );

	Log::Debug(OSS() << "Read gradient name                    : " << gradientDesc.name_);
	Log::Debug(OSS() << "Read gradient name provided           : " << WrapperProvidedAsString[gradientDesc.provided_]);




	WrapperFunctionDescription hessianDesc;
	xmlNodePtr hessianElt = XML::FindElementByName( descriptionElt, XMLTag_hessian );
	hessianDesc.name_     = XML::GetNodeValue( hessianElt );

	String hessianProvided_  = XML::GetAttributeByName( hessianElt, XMLTag_provided );
	if (hessianProvided_ == XMLAttr_yes) hessianDesc.provided_ = WrapperSymbolProvided::YES;
	else if (hessianProvided_ == XMLAttr_no) hessianDesc.provided_ = WrapperSymbolProvided::NO;
	else if (hessianProvided_.empty()) hessianDesc.provided_ = WrapperSymbolProvided::NO;
	else {
	  /* We should never go here if the wrapper file has been validated according to the DTD */
	  throw WrapperFileParsingException(HERE) << "Unknown " << XMLTag_provided << " attribute (" << hessianProvided_
						  << ") for variable in wrapper file " << pathToFile
						  << " at line " << xmlGetLineNo(hessianElt);
	}
	data_.setHessianDescription( hessianDesc );

	Log::Debug(OSS() << "Read hessian  name                    : " << hessianDesc.name_);
	Log::Debug(OSS() << "Read hessian  name provided           : " << WrapperProvidedAsString[hessianDesc.provided_]);



	xmlNodePtr externalCodeElt = XML::FindElementByName( wrapperElt, XMLTag_external_code );
	xmlNodePtr dataElt         = XML::FindElementByName( externalCodeElt, XMLTag_data );
	WrapperData::FileListType fileList;
	if (dataElt && dataElt->children) {
	  for (xmlNodePtr current = dataElt->children; current; current = current->next) {
	    if (XML::IsElement(current, XMLTag_file)) {
	      WrapperDataFile file;

	      xmlNodePtr nameElt  = XML::FindElementByName( current, XMLTag_name );
	      if (nameElt) file.name_ = XML::GetNodeValue( nameElt );
	      
	      xmlNodePtr pathElt  = XML::FindElementByName( current, XMLTag_path );
	      file.path_          = XML::GetNodeValue( pathElt );
	      
	      xmlNodePtr substElt = XML::FindElementByName( current, XMLTag_subst );
	      if (substElt) file.subst_ = XML::GetNodeValue( substElt );
	      
	      file.id_            = XML::GetAttributeByName( current, XMLTag_id );
	      
	      String type_        = XML::GetAttributeByName( current, XMLTag_type );
	      if (type_ == XMLAttr_in) file.type_ = WrapperDataFileType::IN;
	      else if (type_ == XMLAttr_out) file.type_ = WrapperDataFileType::OUT;
	      else {
		/* We should never go here if the wrapper file has been validated according to the DTD */
		throw WrapperFileParsingException(HERE) << "Unknown type (" << type_
							<< ") for file in wrapper file " << pathToFile
							<< " at line " << xmlGetLineNo(current);
	      }
	      
	      Log::Debug(OSS() << "Read data file id                 : " << file.id_);
	      Log::Debug(OSS() << "Read data file name               : " << file.name_);
	      Log::Debug(OSS() << "Read data file path               : " << file.path_);
	      Log::Debug(OSS() << "Read data file subst              : " << file.subst_);
	      Log::Debug(OSS() << "Read data file type               : " << WrapperListElementTypeAsString[file.type_]);

	      fileList.add( file );
	    }
	  }
	}
	data_.setFileList( fileList );



	WrapperParameter parameters;
	xmlNodePtr wrapModeElt = XML::FindElementByName( externalCodeElt, XMLTag_wrap_mode );
	String  wrapType_      = XML::GetAttributeByName( wrapModeElt, XMLTag_type );
	if (wrapType_ == XMLAttr_static_link) parameters.mode_ = WrapperMode::STATICLINK;
	else if (wrapType_ == XMLAttr_dynamic_link) parameters.mode_ = WrapperMode::DYNAMICLINK;
	else if (wrapType_ == XMLAttr_fork) parameters.mode_ = WrapperMode::FORK;
	else {
	  /* We should never go here if the wrapper file has been validated according to the DTD */
	  throw WrapperFileParsingException(HERE) << "Unknown type (" << wrapType_
						  << ") for " << XMLTag_wrap_mode << " in wrapper file " << pathToFile
						  << " at line " << xmlGetLineNo(wrapModeElt);
	}

	String  wrapState_    = XML::GetAttributeByName( wrapModeElt, XMLTag_state );
	if (wrapState_ == XMLAttr_shared) parameters.state_ = WrapperState::SHARED;
	else if (wrapState_ == XMLAttr_specific) parameters.state_ = WrapperState::SPECIFIC;
	else if (wrapState_.empty()) parameters.state_ = WrapperState::SHARED;
	else {
	  /* We should never go here if the wrapper file has been validated according to the DTD */
	  throw WrapperFileParsingException(HERE) << "Unknown state (" << wrapState_
						  << ") for " << XMLTag_wrap_mode << " in wrapper file " << pathToFile
						  << " at line " << xmlGetLineNo(wrapModeElt);
	}


	xmlNodePtr inDataTransferElt = XML::FindElementByName( wrapModeElt, XMLTag_in_data_transfer );
	String  inMode_              = XML::GetAttributeByName( inDataTransferElt, XMLTag_mode );
	if (inMode_ == XMLAttr_files) parameters.in_ = WrapperDataTransfer::FILES;
	else if (inMode_ == XMLAttr_pipe) parameters.in_ = WrapperDataTransfer::PIPE;
	else if (inMode_ == XMLAttr_arguments) parameters.in_ = WrapperDataTransfer::ARGUMENTS;
	else if (inMode_ == XMLAttr_socket) parameters.in_ = WrapperDataTransfer::SOCKET;
	else if (inMode_ == XMLAttr_corba) parameters.in_ = WrapperDataTransfer::CORBA;
	else {
	  /* We should never go here if the wrapper file has been validated according to the DTD */
	  throw WrapperFileParsingException(HERE) << "Unknown mode (" << inMode_
						  << ") for " << XMLTag_in_data_transfer << " in wrapper file " << pathToFile
						  << " at line " << xmlGetLineNo(inDataTransferElt);
	}


	xmlNodePtr outDataTransferElt = XML::FindElementByName( wrapModeElt, XMLTag_out_data_transfer );
	String  outMode_              = XML::GetAttributeByName( outDataTransferElt, XMLTag_mode );
	if (outMode_ == XMLAttr_files) parameters.out_ = WrapperDataTransfer::FILES;
	else if (outMode_ == XMLAttr_pipe) parameters.out_ = WrapperDataTransfer::PIPE;
	else if (outMode_ == XMLAttr_arguments) parameters.out_ = WrapperDataTransfer::ARGUMENTS;
	else if (outMode_ == XMLAttr_socket) parameters.out_ = WrapperDataTransfer::SOCKET;
	else if (outMode_ == XMLAttr_corba) parameters.out_ = WrapperDataTransfer::CORBA;
	else {
	  /* We should never go here if the wrapper file has been validated according to the DTD */
	  throw WrapperFileParsingException(HERE) << "Unknown mode (" << outMode_
						  << ") for " << XMLTag_out_data_transfer << " in wrapper file " << pathToFile
						  << " at line " << xmlGetLineNo(outDataTransferElt);
	}


	xmlNodePtr commandElt = XML::FindElementByName( externalCodeElt, XMLTag_command );
	parameters.command_   = XML::GetNodeValue( commandElt );

	data_.setParameters( parameters );

	Log::Debug(OSS() << "Read wrapper mode                     : " << WrapperConfigurationModeAsString[parameters.mode_]);
	Log::Debug(OSS() << "Read wrapper state                    : " << WrapperConfigurationStateAsString[parameters.state_]);
	Log::Debug(OSS() << "Read wrapper input  transfer mode     : " << WrapperDataTransferModeAsString[parameters.in_]);
	Log::Debug(OSS() << "Read wrapper output transfer mode     : " << WrapperDataTransferModeAsString[parameters.out_]);
	Log::Debug(OSS() << "Read wrapper command                  : " << parameters.command_);


	Log::Debug(OSS() << "file " << pathToFile << " parsed");
      }


#else

      /* Standard Initialization */
      void WrapperFile::init() const
	throw(WrapperFileParsingException)
      {
	// Nothing to do
      }

      /* Standard finalization */
      void WrapperFile::done() const
      {
	// Nothing to do
      }

      /* Standard parsing function */
      void WrapperFile::parseFile(const FileName & pathToFile)
	throw(WrapperFileParsingException)
      {
	// TODO
	throw WrapperFileParsingException("In void WrapperFile::parseFile(const FileName & pathToFile). Not yet implemented.");
      }

#endif


      /* Description file path accessor */
      void WrapperFile::setDescriptionFilePath(const FileName & path)
      {
	descriptionFilePath_ = path;
      }

      /* Description file path accessor */
      FileName WrapperFile::getDescriptionFilePath() const
      {
	return descriptionFilePath_;
      }

      /* Wrapper data accessor */
      void WrapperFile::setWrapperData(const WrapperData & data)
      {
	data_ = data;
      }

      /* Wrapper data accessor */
      const WrapperData & WrapperFile::getWrapperData() const
      {
	return data_;
      }





      /*
       * Find the path of a wrapper file from its name.
       */
      FileName WrapperFile::FindWrapperPathByName(const String & name)
	throw(NoWrapperFileFoundException)
      {
	// Append the extension to the name
	FileName wrapperFileName = name + WrapperFile::extension_;
	Log::Debug( OSS() << "Transform '" << name << "' into '" << wrapperFileName << "'" );

	// Get the directory list...
	std::vector<FileName> directoryList = Path::GetWrapperDirectoryList();

	// ... and search it for the file
	FileName wrapperFullPath;
	try {
	  wrapperFullPath = Path::FindFileByNameInDirectoryList(wrapperFileName, directoryList);

	}
	catch (FileNotFoundException & ex) {
	  throw NoWrapperFileFoundException(HERE) << ex.str();
	}

	return wrapperFullPath;

      } /* end findWrapperPathByName */



      /*
       * Build a wrapper from a name
       */
      WrapperFile WrapperFile::FindWrapperByName(const String & name)
	throw(NoWrapperFileFoundException)
      {
	// We need to constructs object in the body because of some useful temporary
	// It also speeds the creation of the object because the wrapper file is parsed only once
	FileName wrapperPath;
	  
	// Find the path of the wrapper file that describes the numerical
	// math function interface
	Log::Debug(OSS() << "Try loading a wrapper for function '" << name << "'");
	wrapperPath = WrapperFile::FindWrapperPathByName( name );
	  
	// Open the wrapper file and get the name of the library and the names of
	// the function, its gradient and its hessian if defined
	// If the gradient or the hessian are not defined, ask for default ones
	// Then create the actual functions
	// Get also the data read inside the wrapper file and pass them to the wrapper
	WrapperFile wrapperFile( wrapperPath );
	wrapperFile.setName( name );

	return wrapperFile;
      } /* end FindWrapperByName */


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
