//                                               -*- C++ -*-
/**
 *  @file  WrapperData.cxx
 *  @brief This class declares the wrapper data that are exchanged with the platform
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-29 18:54:45 +0100 (mer 29 oct 2008) $
 *  Id:      $Id: WrapperData.cxx 990 2008-10-29 17:54:45Z dutka $
 */
#include <iostream>               // for std::ostream
#include <cstring>                // for std::strncpy
#include <algorithm>              // for std::copy
#include "OSS.hxx"
#include "WrapperCommon.h"
#include "WrapperData.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Func
    {


      WrapperFunctionDescription::WrapperFunctionDescription()
	: name_(),
	  provided_(WrapperSymbolProvided::NO)
      {
	// Nothing to do
      }

      String WrapperFunctionDescription::str() const
      {
	OSS oss;
	oss << "name="      << name_
	    << " provided=" << WrapperProvidedAsString[provided_];
	return oss;
      }




      WrapperDataFile::WrapperDataFile()
	: id_(),
	  path_(),
	  subst_(),
	  type_(WrapperDataFileType::IN)
      {
	// Nothing to do
      }

      String WrapperDataFile::str() const
      {
	OSS oss;
	oss << "id="     << id_
	    << " name="  << name_
	    << " path="  << path_
	    << " subst=" << subst_
	    << " type="  << WrapperListElementTypeAsString[type_];
	return oss;
      }





      WrapperDataVariable::WrapperDataVariable()
	: id_(),
	  comment_(),
	  unit_(),
	  regexp_(),
	  format_(),
	  type_(WrapperDataVariableType::IN),
	  gradient_(WrapperComputedGradient::NO)
      {
	// Nothing to do
      }


      String WrapperDataVariable::str() const
      {
	OSS oss;
	oss << "id="        << id_
	    << " comment="  << comment_
	    << " unit="     << unit_
	    << " regexp="   << regexp_
	    << " format="   << format_
	    << " type="     << WrapperListElementTypeAsString[type_]
	    << " gradient=" << WrapperProvidedAsString[gradient_];
	return oss;
      }




      WrapperParameter::WrapperParameter()
	: state_(WrapperState::SHARED),
	  mode_(WrapperMode::STATICLINK),
	  in_(WrapperDataTransfer::FILES),
	  out_(WrapperDataTransfer::FILES),
	  command_()
      {
	// Nothing to do
      }


      String WrapperParameter::str() const
      {
	OSS oss;
	oss << "state="  << WrapperConfigurationStateAsString[state_]
	    << " mode="  << WrapperConfigurationModeAsString[mode_]
	    << " in="    << WrapperDataTransferModeAsString[in_]
	    << " out="   << WrapperDataTransferModeAsString[out_];
	return oss;
      }


      CLASSNAMEINIT(WrapperData);

      /* Default constructor */
      WrapperData::WrapperData()
	: Object(),
	  libraryPath_(),
	  function_(),
	  gradient_(),
	  hessian_(),
	  p_fileList_(new WrapperData::FileListType),
	  p_variableList_(new WrapperData::VariableListType),
	  p_parameters_(new WrapperParameter)
      {
	// Nothing to do
      }



      /* String converter */
      String WrapperData::str() const
      {	OSS oss;
	oss << "libraryPath=" << libraryPath_
	    << " function={"  << function_
	    << "} gradient={" << gradient_
	    << "} hessian={"  << hessian_
	    << "} fileList=[";

	std::copy( p_fileList_->begin(), p_fileList_->end(), OSS_iterator<WrapperDataFile>(oss, ", ") );

	oss << "] variableList=[";

	std::copy( p_variableList_->begin(), p_variableList_->end(), OSS_iterator<WrapperDataVariable>(oss, ", ") );

	oss << "] parameters={" << *p_parameters_
	    << "}";
	return oss;

      }


      /* Library path accessor */
      void WrapperData::setLibraryPath(const FileName & path)
      {
	libraryPath_ = path;
      }

      /* Library path accessor */
      FileName WrapperData::getLibraryPath() const
      {
	return libraryPath_;
      }

      /* Function description accessor */
      void WrapperData::setFunctionDescription(const WrapperFunctionDescription & funcDescription)
      {
	function_ = funcDescription;
      }

      /* Function description accessor */
      WrapperFunctionDescription WrapperData::getFunctionDescription() const
      {
	return function_;
      }

      /* Gradient description accessor */
      void WrapperData::setGradientDescription(const WrapperFunctionDescription & gradDescription)
      {
	gradient_ = gradDescription;
      }

      /* Gradient description accessor */
      WrapperFunctionDescription WrapperData::getGradientDescription() const
      {
	return gradient_;
      }

      /* Hessian description accessor */
      void WrapperData::setHessianDescription(const WrapperFunctionDescription & hessDescription)
      {
	hessian_ = hessDescription;
      }

      /* Hessian description accessor */
      WrapperFunctionDescription WrapperData::getHessianDescription() const
      {
	return hessian_;
      }

      /* Accessor */
      void WrapperData::setFileList(const WrapperData::FileListType & fileList)
      {
	p_fileList_.reset(new WrapperData::FileListType(fileList));
      }



      /* Accessor */
      const WrapperData::FileListType & WrapperData::getFileList() const
      {
	return *p_fileList_;
      }


      /* Conversion method for C interface */
      struct WrapperFileList * WrapperData::getNewFileListForCInterface() const
      {
	struct WrapperFileList * fileList = 0;

	// we convert the data into the exchange format
	struct WrapperFileList ** previousElementPointer = & fileList;
	FileListType::const_iterator fileList_iterator;
	for(fileList_iterator  = getFileList().begin();
	    fileList_iterator != getFileList().end();
	    fileList_iterator++) {
	  WrapperDataFile file = *fileList_iterator;

	  struct WrapperFileList * p_linkOnFile    = new WrapperFileList;
	  p_linkOnFile->next_ = 0;
	  p_linkOnFile->file_ = new WrapperFileListElement;

	  p_linkOnFile->file_->id_ = new char[file.id_.size() + 1];
	  strncpy(p_linkOnFile->file_->id_, file.id_.c_str(), file.id_.size() + 1);

	  p_linkOnFile->file_->name_ = new char[file.name_.size() + 1];
	  strncpy(p_linkOnFile->file_->name_, file.name_.c_str(), file.name_.size() + 1);

	  p_linkOnFile->file_->path_ = new char[file.path_.size() + 1];
	  strncpy(p_linkOnFile->file_->path_, file.path_.c_str(), file.path_.size() + 1);

	  p_linkOnFile->file_->subst_ = new char[file.subst_.size() + 1];
	  strncpy(p_linkOnFile->file_->subst_, file.subst_.c_str(), file.subst_.size() + 1);

	  p_linkOnFile->file_->type_ = file.type_;

	  // we insert the element into the list...
	  *previousElementPointer = p_linkOnFile;
	  
	  // ... then we jump to the next element
	  previousElementPointer = & p_linkOnFile->next_;

	} /* end for */

	return fileList;
      }


      /* Frees the memory allocated by getNewFileListForCInterface() method */
      void WrapperData::freeFileListForCInterface(const struct WrapperFileList * fileList) const
      {
	const struct WrapperFileList * currentFileElement = fileList;
	while (currentFileElement) {
	  const struct WrapperFileList * nextElement = currentFileElement->next_;
	  delete [] currentFileElement->file_->id_;
	  delete [] currentFileElement->file_->name_;
	  delete [] currentFileElement->file_->path_;
	  delete [] currentFileElement->file_->subst_;
	  delete currentFileElement->file_;
	  delete currentFileElement;
	  currentFileElement = nextElement;
	}
      }



      /* Accessor */
      void WrapperData::setVariableList(const WrapperData::VariableListType & variableList)
      {
	p_variableList_.reset(new WrapperData::VariableListType(variableList));
      }



      /* Accessor */
      const WrapperData::VariableListType & WrapperData::getVariableList() const
      {
	return *p_variableList_;
      }


      /* Conversion method for C interface */
      struct WrapperVariableList * WrapperData::getNewVariableListForCInterface() const
      {
	struct WrapperVariableList * variableList = 0;

	// we convert the data into the exchange format
	struct WrapperVariableList ** previousElementPointer = & variableList;
	VariableListType::const_iterator variableList_iterator;
	for(variableList_iterator  = getVariableList().begin();
	    variableList_iterator != getVariableList().end();
	    variableList_iterator++) {
	  WrapperDataVariable variable = *variableList_iterator;

	  struct WrapperVariableList * p_linkOnVariable    = new WrapperVariableList;
	  p_linkOnVariable->next_ = 0;
	  p_linkOnVariable->variable_ = new WrapperVariableListElement;

	  p_linkOnVariable->variable_->comment_ = new char[variable.comment_.size() + 1];
	  strncpy(p_linkOnVariable->variable_->comment_, variable.comment_.c_str(), variable.comment_.size() + 1);

	  p_linkOnVariable->variable_->unit_ = new char[variable.unit_.size() + 1];
	  strncpy(p_linkOnVariable->variable_->unit_, variable.unit_.c_str(), variable.unit_.size() + 1);

	  p_linkOnVariable->variable_->regexp_ = new char[variable.regexp_.size() + 1];
	  strncpy(p_linkOnVariable->variable_->regexp_, variable.regexp_.c_str(), variable.regexp_.size() + 1);

	  p_linkOnVariable->variable_->format_ = new char[variable.format_.size() + 1];
	  strncpy(p_linkOnVariable->variable_->format_, variable.format_.c_str(), variable.format_.size() + 1);

	  p_linkOnVariable->variable_->type_ = variable.type_;

	  p_linkOnVariable->variable_->gradient_ = variable.gradient_;

	  // we insert the element into the list...
	  *previousElementPointer = p_linkOnVariable;
	  
	  // ... then we jump to the next element
	  previousElementPointer = & p_linkOnVariable->next_;

	} /* end for */

	return variableList;
      }


      /* Frees the memory allocated by getNewVariableListForCInterface() method */
      void WrapperData::freeVariableListForCInterface(const struct WrapperVariableList * variableList) const
      {
	const struct WrapperVariableList * currentVariableElement = variableList;
	while (currentVariableElement) {
	  const struct WrapperVariableList * nextElement = currentVariableElement->next_;
	  delete [] currentVariableElement->variable_->comment_;
	  delete [] currentVariableElement->variable_->unit_;
	  delete [] currentVariableElement->variable_->regexp_;
	  delete [] currentVariableElement->variable_->format_;
	  delete currentVariableElement->variable_;
	  delete currentVariableElement;
	  currentVariableElement = nextElement;
	}
      }



      /* Accessor */
      void WrapperData::setParameters(const WrapperParameter & parameters)
      {
	p_parameters_.reset(new WrapperParameter(parameters));
      }

      const WrapperParameter & WrapperData::getParameters() const
      {
	return *p_parameters_;
      }
      
      /* Conversion method for C interface */
      struct WrapperConfiguration * WrapperData::getNewParametersForCInterface() const
      {
	struct WrapperConfiguration * parameters = new WrapperConfiguration;

	parameters->state_ = p_parameters_->state_;
	parameters->mode_  = p_parameters_->mode_;
	parameters->in_    = p_parameters_->in_;
	parameters->out_   = p_parameters_->out_;

	parameters->command_ = new char[p_parameters_->command_.size() + 1];
	strncpy(parameters->command_, p_parameters_->command_.c_str(), p_parameters_->command_.size() + 1);

	return parameters;
      }
      
      /* Frees the memory allocated by getNewParametersForCInterface() method */
      void WrapperData::freeParametersForCInterface(const struct WrapperConfiguration * parameters) const
      {
	delete [] parameters->command_;
	delete parameters;
      }
      

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
