//                                               -*- C++ -*-
/**
 *  @file  NumericalMathHessianImplementation.hxx
 *  @brief Abstract top-level class for all numerical math hessian implementations
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: NumericalMathHessianImplementation.hxx 995 2008-10-31 10:52:04Z dutka $
 */
#ifndef OPENTURNS_NUMERICALMATHHESSIANIMPLEMENTATION_HXX
#define OPENTURNS_NUMERICALMATHHESSIANIMPLEMENTATION_HXX

#include "PersistentObject.hxx"
#include "NumericalPoint.hxx"
#include "Exception.hxx"
#include "Description.hxx"
#include "SymmetricTensor.hxx"
#include "Pointer.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {


      /**
       * @class NumericalMathHessianImplementation
       *
       * This class offers an abstract interface for the implementation
       * of an real numerical mathematical hessian into the platform.
       */
      class NumericalMathHessianImplementation 
	: public Common::PersistentObject
      {
	CLASSNAME;
      public:

	typedef Common::Pointer<NumericalMathHessianImplementation> Implementation;
	typedef Type::Description                                   Description;
	typedef Type::SymmetricTensor                               SymmetricTensor;
	typedef Type::NumericalPoint                                NumericalPoint;
	typedef Common::InvalidArgumentException                    InvalidArgumentException;
	typedef Common::InternalException                           InternalException;
	typedef Common::StorageManager                              StorageManager;


	/** Default constructor */
	NumericalMathHessianImplementation();

	/** Virtual constructor */
	virtual NumericalMathHessianImplementation * clone() const;

	/** Comparison operator */
	Bool operator ==(const NumericalMathHessianImplementation & other) const;

	/** String converter */
	virtual String str() const;

	/** Description Accessor */
	void setDescription(const Description & description);
	Description getDescription() const;

	/** Test for actual implementation */
	virtual Bool isActualImplementation() const;



	/* Here is the interface that all derived class must implement */
	
	/** Hessian method */
	virtual SymmetricTensor hessian(const NumericalPoint & in) const
	  throw(InvalidArgumentException,InternalException);

	/** Accessor for input point dimension */
	virtual UnsignedLong getInputNumericalPointDimension() const
	  throw(InternalException);

	/** Accessor for output point dimension */
	virtual UnsignedLong getOutputNumericalPointDimension() const
	  throw(InternalException);

	/** Get the number of calls to operator() */
	UnsignedLong getCallsNumber() const;

 	/** Method save() stores the object through the StorageManager */
	void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(const StorageManager::Advocate & adv);

      protected:

	/** Number of calls since the construction */
	mutable UnsignedLong callsNumber_;

      private:

 	/** The description of all components */
	Description description_;


      }; /* class NumericalMathHessianImplementation */


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_NUMERICALMATHHESSIANIMPLEMENTATION_HXX */
