//                                               -*- C++ -*-
/**
 *  @file  NumericalMathGradientImplementation.hxx
 *  @brief Abstract top-level class for all numerical math gradient implementations
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: NumericalMathGradientImplementation.hxx 995 2008-10-31 10:52:04Z dutka $
 */
#ifndef OPENTURNS_NUMERICALMATHGRADIENTIMPLEMENTATION_HXX
#define OPENTURNS_NUMERICALMATHGRADIENTIMPLEMENTATION_HXX

#include "PersistentObject.hxx"
#include "NumericalPoint.hxx"
#include "Exception.hxx"
#include "Description.hxx"
#include "Matrix.hxx"
#include "Tensor.hxx"
#include "Pointer.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {


      /**
       * @class NumericalMathGradientImplementation
       *
       * This class offers an abstract interface for the implementation
       * of an real numerical mathematical Gradient into the platform.
       */
      class NumericalMathGradientImplementation 
	: public Common::PersistentObject
      {
	CLASSNAME;
      public:

	typedef Common::Pointer<NumericalMathGradientImplementation> Implementation;
	typedef Type::Description                                    Description;
	typedef Type::NumericalPoint                                 NumericalPoint;
	typedef Type::Matrix                                         Matrix;
	typedef Type::Tensor                                         Tensor;
	typedef Common::InvalidArgumentException                     InvalidArgumentException;
	typedef Common::InternalException                            InternalException;
	typedef Common::StorageManager                               StorageManager;


	/** Default constructor */
	NumericalMathGradientImplementation();

	/** The Gradient that actually realizes assignment between objects */
    
	/** Virtual constructor */
	virtual NumericalMathGradientImplementation * clone() const;

	/** Comparison operator */
	Bool operator ==(const NumericalMathGradientImplementation & other) const;

	/** String converter */
	virtual String str() const;

	/** Description Accessor */
	void setDescription(const Description & description);
	Description getDescription() const;

	/** Test for actual implementation */
	virtual Bool isActualImplementation() const;



	/* Here is the interface that all derived class must implement */
	
	/** Gradient method */
	virtual Matrix gradient(const NumericalPoint & in) const
	  throw(InvalidArgumentException, InternalException);

	/** Accessor for input point dimension */
	virtual UnsignedLong getInputNumericalPointDimension() const
	  throw(InternalException);

	/** Accessor for output point dimension */
	virtual UnsignedLong getOutputNumericalPointDimension() const
	  throw(InternalException);

	/** Get the number of calls to operator() */
	UnsignedLong getCallsNumber() const;

 	/** Method save() stores the object through the StorageManager */
	void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(const StorageManager::Advocate & adv);

      protected:

	/** Number of calls since the construction */
	mutable UnsignedLong callsNumber_;

      private:

 	/** The description of all components */
	Description description_;


      }; /* class NumericalMathGradientImplementation */


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_NUMERICALMATHGRADIENTIMPLEMENTATION_HXX */
