//                                               -*- C++ -*-
/**
 * @file  MethodBoundNumericalMathEvaluationImplementation.hxx
 * @brief This class gives a implementation for object's methods so they can be used in NumericalMathFunctions
 *
 * (C) Copyright 2005-2006 EDF
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * \author $LastChangedBy: dutka $
 * \date   $LastChangedDate: 2006-04-07 12:36:59 +0200 (Fri, 07 Apr 2006) $
 */

#ifndef OPENTURNS_METHODBOUNDNUMERICALMATHEVALUATIONIMPLEMENTATION_HXX
#define OPENTURNS_METHODBOUNDNUMERICALMATHEVALUATIONIMPLEMENTATION_HXX

#include "NumericalMathEvaluationImplementation.hxx"
#include "NumericalPoint.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      struct NO_MATCH_ON_ReturnTypeAdapter {};

      template <typename Tp_> struct ReturnTypeAdapter
      {
	typedef NO_MATCH_ON_ReturnTypeAdapter Type_;
      };

      template <> struct ReturnTypeAdapter< NumericalScalar > {
	typedef NumericalScalar Type_;
	static inline Type::NumericalPoint toPoint( NumericalScalar val ) { return Type::NumericalPoint(1,val) ; }	
      };

      template <> struct ReturnTypeAdapter< Type::NumericalPoint > {
	typedef Type::NumericalPoint Type_;
	static inline Type::NumericalPoint toPoint( const Type::NumericalPoint & val ) { return val ; }	
      };



      struct NO_MATCH_ON_ArgumentTypeAdapter {};

      template <typename Tp_> struct ArgumentTypeAdapter
      {
	typedef NO_MATCH_ON_ArgumentTypeAdapter Type_;
      };

      template <> struct ArgumentTypeAdapter< NumericalScalar > {
	typedef const NumericalScalar Type_;
	static inline const NumericalScalar fromPoint( const Type::NumericalPoint & val ) { return val[0] ; }
      };

      template <> struct ArgumentTypeAdapter< Type::NumericalPoint > {
	typedef const Type::NumericalPoint & Type_;
	static inline Type::NumericalPoint fromPoint( const Type::NumericalPoint & val ) { return val ; }
      };



      template <typename ReturnType_, typename ArgumentType_>
      struct MethodAdapter : public ReturnTypeAdapter<ReturnType_>, public ArgumentTypeAdapter<ArgumentType_>
      {
	typedef typename ArgumentTypeAdapter<ArgumentType_>::Type_ ArgumentType;
	typedef typename ReturnTypeAdapter<ReturnType_>::Type_     ReturnType;
      };






      /**
       * @class MethodBoundNumericalMathEvaluationImplementation
       *
       * This class gives a implementation for object's methods so they can be used in NumericalMathFunctions
       */
      template <typename EvaluableObject, typename ReturnType_, typename ArgumentType_>
      class MethodBoundNumericalMathEvaluationImplementation
	: public NumericalMathEvaluationImplementation
      {
      public:

	typedef NumericalMathEvaluationImplementation::NumericalPoint           NumericalPoint;
	typedef NumericalMathEvaluationImplementation::InvalidArgumentException InvalidArgumentException;
	typedef NumericalMathEvaluationImplementation::InternalException        InternalException;


	typedef typename MethodAdapter<ReturnType_, ArgumentType_>::ReturnType (EvaluableObject::*EvaluationMethod) (typename MethodAdapter<ReturnType_, ArgumentType_>::ArgumentType) const;



	/** Default constructor */
	MethodBoundNumericalMathEvaluationImplementation( const EvaluableObject & obj, EvaluationMethod method ) 
	  : obj_(obj),
	    method_(method)
	{
	  // Nothing to do
	}


	/** Virtual constructor */
	virtual MethodBoundNumericalMathEvaluationImplementation * clone() const
	{
	  return new MethodBoundNumericalMathEvaluationImplementation(*this);
	}


	/** Comparison operator */
	Bool operator ==(const MethodBoundNumericalMathEvaluationImplementation & other) const
	{
	  return true;
	}


	/** String converter */
	virtual String str() const
	{
	  OSS oss;
	  oss << "class=MethodBoundNumericalMathEvaluationImplementation name=" << getName();
	  return oss;
	}


	/** Test for actual implementation */
	virtual Bool isActualImplementation() const
	{
	  return true;
	}


	/* Here is the interface that all derived class must implement */
	
	/** Operator () */
	virtual NumericalPoint operator() (const NumericalPoint & in) const
	  throw(InvalidArgumentException,InternalException)
	{
	  return ReturnTypeAdapter<ReturnType_>::toPoint( ( obj_.*method_ ) ( ArgumentTypeAdapter<ArgumentType_>::fromPoint( in ) ) );
	}

	/** Accessor for input point dimension */
	virtual UnsignedLong getInputNumericalPointDimension() const
	  throw(InternalException)
	{
	  return obj_.getDimension();
	}

	/** Accessor for output point dimension */
	virtual UnsignedLong getOutputNumericalPointDimension() const
	  throw(InternalException)
	{
	  return obj_.getDimension();
	}


 	/** Method save() stores the object through the StorageManager
	 *  The inherited method is sufficient as we do not have any
         *  attribute
         */

	/** Method load() reloads the object from the StorageManager
	 *  The inherited method is sufficient as we do not have any
         *  attribute
         */

      protected:

    
      private:
	const EvaluableObject & obj_;
	EvaluationMethod method_;

      }; /* class MethodBoundNumericalMathEvaluationImplementation */





 
      template <typename EvaluableObject, typename ReturnType_, typename ArgumentType_>
      MethodBoundNumericalMathEvaluationImplementation<EvaluableObject, ReturnType_, ArgumentType_> *
      bindMethod (const EvaluableObject & obj,
		  typename MethodBoundNumericalMathEvaluationImplementation<EvaluableObject, ReturnType_, ArgumentType_>::EvaluationMethod method )
      {
	return new MethodBoundNumericalMathEvaluationImplementation<EvaluableObject, ReturnType_, ArgumentType_>( obj, method );
      }



   } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_METHODBOUNDNUMERICALMATHEVALUATIONIMPLEMENTATION_HXX */
