//                                               -*- C++ -*-
/**
 *  @file  ComputedNumericalMathGradientImplementation.hxx
 *  @brief Abstract top-level class for all ComputedNumerical math functions
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-17 14:26:02 +0200 (ven 17 oct 2008) $
 *  Id:      $Id: ComputedNumericalMathGradientImplementation.hxx 974 2008-10-17 12:26:02Z dutka $
 */
#ifndef OPENTURNS_COMPUTEDNUMERICALMATHGRADIENTIMPLEMENTATION_HXX
#define OPENTURNS_COMPUTEDNUMERICALMATHGRADIENTIMPLEMENTATION_HXX

#include "NumericalMathGradientImplementation.hxx"
#include "Pointer.hxx"
#include "NumericalPoint.hxx"
#include "Matrix.hxx"
#include "WrapperFile.hxx"
#include "WrapperObject.hxx"
#include "Exception.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {


      /**
       * @class ComputedNumericalMathGradientImplementation
       *
       * This class acts like an real numerical mathematical gradient
       * into the platform. It realizes the computation of the image vector
       * of a vector (aka NumericalPoint) through the function. The
       * function may be loaded in an external way through a wrapper.
       */
      class ComputedNumericalMathGradientImplementation 
	: public NumericalMathGradientImplementation
      {
	CLASSNAME;

      public:

	/* Some typedefs to ease reading */
	typedef Common::WrapperInternalException                              WrapperInternalException;
	typedef NumericalMathGradientImplementation::InvalidArgumentException InvalidArgumentException;
	typedef NumericalMathGradientImplementation::InternalException        InternalException;
	typedef Common::Pointer<WrapperObject>                                InternalGradient;
	typedef NumericalMathGradientImplementation::NumericalPoint           NumericalPoint;
	typedef NumericalMathGradientImplementation::Matrix                   Matrix;

	/** Default constructor */
	ComputedNumericalMathGradientImplementation(const String & name, const WrapperFile & file, void * p_state = 0)
	  throw(WrapperInternalException);

	/** Copy constructor */
	ComputedNumericalMathGradientImplementation(const ComputedNumericalMathGradientImplementation & other)
	  throw(WrapperInternalException);

	/** Virtual constructor */
	virtual ComputedNumericalMathGradientImplementation * clone() const;

	/** Destructor */
	virtual ~ComputedNumericalMathGradientImplementation();

	/** Comparison operator */
	Bool operator ==(const ComputedNumericalMathGradientImplementation & other) const;

	/** String converter */
	String str() const;



	/* Here is the interface that all derived class must implement */
	
	/** Gradient method */
	virtual Matrix gradient(const NumericalPoint & in) const
	  throw(InvalidArgumentException,InternalException);

	/** Accessor for input point dimension */
	virtual UnsignedLong getInputNumericalPointDimension() const
	  throw(InternalException);

	/** Accessor for output point dimension */
	virtual UnsignedLong getOutputNumericalPointDimension() const
	  throw(InternalException);

 	/** Method save() stores the object through the StorageManager */
	void save(const StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(const StorageManager::Advocate & adv);

      protected:

	/** The function object that is linked to the external code */
	InternalGradient p_gradient_;

	/** State management */
    	void * getState() const;

      private:
	ComputedNumericalMathGradientImplementation() {};
	friend class Common::Factory<ComputedNumericalMathGradientImplementation>;

	/** A pointer to an internal structure that saves the state of the function into the wrapper */
	void * p_state_;

	/** A flag that is true when the gradient shares its internal state with its related function */
	Bool commonState_;

      }; /* class ComputedNumericalMathGradientImplementation */


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_COMPUTEDNUMERICALMATHGRADIENTIMPLEMENTATION_HXX */
