//                                               -*- C++ -*-
/**
 *  @file  ComputedNumericalMathGradientImplementation.cxx
 *  @brief Abstract top-level class for all distributions
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-31 11:52:04 +0100 (ven 31 oct 2008) $
 *  Id:      $Id: ComputedNumericalMathGradientImplementation.cxx 995 2008-10-31 10:52:04Z dutka $
 */
#include "ComputedNumericalMathGradientImplementation.hxx"
#include "ComputedNumericalMathGradientImplementationFactory.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {

      CLASSNAMEINIT(ComputedNumericalMathGradientImplementation);

      static Common::Factory<ComputedNumericalMathGradientImplementation> RegisteredFactory("ComputedNumericalMathGradientImplementation");

      /* Default constructor */
      ComputedNumericalMathGradientImplementation::ComputedNumericalMathGradientImplementation(const String & name,
											       const WrapperFile & file,
											       void * p_state)
	throw(WrapperInternalException)
	: NumericalMathGradientImplementation(),
	  p_gradient_(0),
	  p_state_(0),
	  commonState_(file.getWrapperData().getParameters().state_ == WRAPPER_SHAREDSTATE)
      {
	setName(name);
	if (file.getWrapperData().getGradientDescription().provided_) {
	  p_gradient_.reset(new WrapperObject( file.getWrapperData().getLibraryPath(),
					       file.getWrapperData().getGradientDescription().name_,
					       file.getWrapperData(),
					       WrapperObject::GRADIENT));
	}

	if (p_gradient_.isNull()) throw WrapperInternalException(HERE) << "Unable to allocate wrapper for gradient (Note: this message is normal if gradient is NOT provided. Check wrapper description file '" << file.getDescriptionFilePath() << ".)";

	// Initialize the state into the wrapper
	p_state_ = commonState_ ? p_state : p_gradient_->createNewState();
      }


      /* Copy constructor */
      ComputedNumericalMathGradientImplementation::ComputedNumericalMathGradientImplementation(const ComputedNumericalMathGradientImplementation & other)
	throw(WrapperInternalException)
	: NumericalMathGradientImplementation(other),
	  p_gradient_(other.p_gradient_),
	  p_state_(0),
	  commonState_(other.commonState_)
      {
	if (p_gradient_.isNull()) throw WrapperInternalException(HERE) << "Unable to allocate wrapper";

	// Initialize the state into the wrapper
	p_state_ = commonState_ ? other.p_state_ : p_gradient_->createNewState();
      }


      /* Virtual constructor */
      ComputedNumericalMathGradientImplementation * ComputedNumericalMathGradientImplementation::clone() const
      {
	return new ComputedNumericalMathGradientImplementation(*this);
      }

      /* Destructor */
      ComputedNumericalMathGradientImplementation::~ComputedNumericalMathGradientImplementation()
      {
	// Delete the state into the wrapper
	if (!commonState_) p_gradient_->deleteState( p_state_ );
      }


      /* Comparison operator */
      Bool ComputedNumericalMathGradientImplementation::operator ==(const ComputedNumericalMathGradientImplementation & other) const
      {
	return true;
      }

  
      /* String converter */
      String ComputedNumericalMathGradientImplementation::str() const {
	OSS oss;
	oss << "class=" << ComputedNumericalMathGradientImplementation::GetClassName()
	    << " name=" << getName();
	return oss;
      }

  
      /* State accessor */
      void * ComputedNumericalMathGradientImplementation::getState() const
      {
	return p_state_;
      }







      /* Here is the interface that all derived class must implement */
	
      /* Gradient method */
      ComputedNumericalMathGradientImplementation::Matrix
      ComputedNumericalMathGradientImplementation::gradient(const NumericalPoint & in) const
	throw(InvalidArgumentException,InternalException)
      {
	try {
	  // Execute the wrapper
	  callsNumber_++;
	  return p_gradient_->gradient( getState(), in );
	}
	catch (InvalidArgumentException & ex) {
	  throw;
	}
 	catch (WrapperInternalException & ex) {
	  throw InternalException(HERE) << ex.str();
	}
      }



      /* Accessor for input point dimension */
      UnsignedLong ComputedNumericalMathGradientImplementation::getInputNumericalPointDimension() const
	throw(InternalException)
      {
	UnsignedLong inputDimension = 0;

	try {
	  inputDimension = p_gradient_->getInNumericalPointDimension(getState());
	}
 	catch (WrapperInternalException & ex) {
	  throw InternalException(HERE) << ex.str();
	}

	return inputDimension;
      }



      /* Accessor for output point dimension */
      UnsignedLong ComputedNumericalMathGradientImplementation::getOutputNumericalPointDimension() const
	throw(InternalException)
      {
	UnsignedLong outputDimension = 0;

	try {
	  outputDimension = p_gradient_->getOutNumericalPointDimension(getState());
	}
 	catch (WrapperInternalException & ex) {
	  throw InternalException(HERE) << ex.str();
	}

	return outputDimension;
      }

      /* Method save() stores the object through the StorageManager */
      void ComputedNumericalMathGradientImplementation::save(const StorageManager::Advocate & adv) const
      {
	NumericalMathGradientImplementation::save(adv);
      }

      /* Method load() reloads the object from the StorageManager */
      void ComputedNumericalMathGradientImplementation::load(const StorageManager::Advocate & adv)
      {
	NumericalMathGradientImplementation::load(adv);
	*this = *static_cast<ComputedNumericalMathGradientImplementation*>(ComputedNumericalMathGradientImplementationFactory::getInstance()->buildObject(getName()));
      }

      
    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
