//                                               -*- C++ -*-
/**
 *  @file  XMLStorageManager.hxx
 *  @brief XMLStorageManager provides an interface for different storage classes
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-29 18:54:45 +0100 (mer 29 oct 2008) $
 *  Id:      $Id: XMLStorageManager.hxx 990 2008-10-29 17:54:45Z dutka $
 */
#ifndef OPENTURNS_XMLSTORAGEMANAGER_HXX
#define OPENTURNS_XMLSTORAGEMANAGER_HXX

#include <iostream>              // for std::ostream
#include <stack>                 // for std::stack
#include "OT.hxx"
#include "StorageManager.hxx"
#include "XMLToolbox.hxx"

#ifndef SWIG
#ifndef XML_SUPPORTED
#error "XML support is mandatory. Check configuration."
#endif
#endif

namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {

      /* Forward declaration of PersistentObject */
      class PersistentObject;

      /**
       * @class XMLStorageManager
       *
       * XMLStorageManager implements a storage manager that
       * drives a XML file
       */

      class XMLStorageManager
	: public StorageManager
      {
	CLASSNAME;

	class VersionList
	{
	  std::vector<UnsignedLong> versions_;
	public:
	  explicit VersionList() : versions_()
	  {
	    versions_.push_back(1);
	  }

	  Bool contains(UnsignedLong v) const
	  {
	    return find(versions_.begin(),versions_.end(),v) != versions_.end();
	  }
	};

	static const VersionList SupportedVersions;

	struct State {
	  XML::Node root_;
	  XML::Node current_;
	  XML::Node sub_;
	  XML::Node node_;
	  State() : root_(0), current_(0), sub_(0), node_(0) {}
	}; /* end struct State */


	class XMLList
	  : public StorageManager::ListImplementation
	{
	  XML::Node nodeList_;
	  XML::Node cursor_;
	  const XMLStorageManager & manager_;
	  UnsignedLong length_;
	  EntityName entity_;
	public:
	  XMLList(const XMLStorageManager & mgr, State state, EntityName entity);
	  virtual StorageManager::ListImplementation * clone() const;
	  virtual UnsignedLong getSize() const;
	  virtual void firstValueToRead();
	  virtual Bool moreValuesToRead();
	  virtual void nextValueToRead();
	  virtual Bool readValue(UnsignedLong & index, NumericalScalar & value);
	  virtual Bool readValue(UnsignedLong & index, NumericalComplex & value);
	  virtual Bool readValue(UnsignedLong & index, InterfaceObject & obj);
	  virtual Bool readValue(UnsignedLong & index, PersistentObject & obj);
	  virtual Bool readValue(UnsignedLong & index, String & st);
	  virtual Bool readValue(UnsignedLong & index, UnsignedLong & value);
	  virtual Bool readValue(UnsignedLong & index, Bool & value);
	  virtual Bool readValue(String & name, NumericalScalar & value);
	  virtual Bool readValue(String & name, NumericalComplex & value);
	  virtual Bool readValue(String & name, InterfaceObject & obj);
	  virtual Bool readValue(String & name, PersistentObject & obj);
	  virtual Bool readValue(String & name, String & st);
	  virtual Bool readValue(String & name, UnsignedLong & value);
	  virtual Bool readValue(String & name, Bool & value);
	  virtual Bool readValue(InterfaceObject & obj);
	  virtual Bool readValue(InterfaceObject & obj, AttributeName attribute, const String & attributeValue);
	  virtual Bool readValue(PersistentObject & obj);
	  virtual Bool readValue(PersistentObject & obj, AttributeName attribute, const String & attributeValue);
	  
	}; /* end class List */

      public:


	typedef std::stack<State> StateStack;

	/** 
	 * Constructor for a XML StorageManager that reads/writes
	 * to a XML file
	 */
        XMLStorageManager(const FileName & filename);

	/**
	 * Virtual constructor
	 */
	virtual XMLStorageManager * clone() const;

	/**
	 * String converter
	 * This method shows human readable information on the
	 * internal state of an object. It is used when streaming
	 * the object or for user information.
	 */
	virtual String str() const;

	/** Filename accessors */
	String getFileName() const;
	void setFileName(const String & fileName);

	/**
	 * This method saves the PersistentObject onto the medium
	 */
	virtual void save(const PersistentObject & obj, bool fromStudy = false);

	/**
	 * This method reloads the PersistentObject from the medium
	 */
	virtual void load(Study & study);



	/** Do some administrative tasks before saving/reloading */
	virtual void initialize(const MethodCall caller);

	/** Do some administrative tasks after saving/reloading */
	virtual void finalize(const MethodCall caller);

	/** Read and create the internal representation */
	virtual void read();

	/** Write the internal representation */
	virtual void write();



      protected:

	/** Query the manager if the version is correct */
	virtual Bool canManageVersion(UnsignedLong version) const;

	/** Methods to control manager internal state */
	virtual void pushState();
	virtual void popState();

	/** List methods */
	virtual List getList(EntityName entity) const;

	/** Methods to write out objects */
	virtual void writeObjectPrologue(const PersistentObject & obj, bool fromStudy = false);
	virtual void writeObjectEpilogue(const PersistentObject & obj);
	
	/** Methods to write out class members */
	virtual void writeAttribute(AttributeName attribute, const String & st);
	virtual void writeAttribute(AttributeName attribute, UnsignedLong ul);

	virtual void writeValue(UnsignedLong index, NumericalScalar value);
	virtual void writeValue(UnsignedLong index, NumericalComplex value);
	virtual void writeValue(UnsignedLong index, const PersistentObject & obj);
	virtual void writeValue(UnsignedLong index, const String & st);
	virtual void writeValue(UnsignedLong index, UnsignedLong value);
	virtual void writeValue(UnsignedLong index, Bool value);
	virtual void writeValue(const String & name, NumericalScalar value);
	virtual void writeValue(const String & name, NumericalComplex value);
	virtual void writeValue(const String & name, const PersistentObject & obj);
	virtual void writeValue(const String & name, const String & st);
	virtual void writeValue(const String & name, UnsignedLong value);
	virtual void writeValue(const String & name, Bool value);
	virtual void writeValue(const PersistentObject & obj);
	virtual void writeValue(const PersistentObject & obj, AttributeName attribute, const String & st);

	/** Methods to read class members */
	virtual void readAttribute(AttributeName attribute, String & st);
	virtual void readAttribute(AttributeName attribute, UnsignedLong & ul);

	virtual void readValue(UnsignedLong & index, NumericalScalar & value);
	virtual void readValue(UnsignedLong & index, NumericalComplex & value);
	virtual void readValue(UnsignedLong & index, InterfaceObject & obj);
	virtual void readValue(UnsignedLong & index, PersistentObject & obj);
	virtual void readValue(UnsignedLong & index, String & st);
	virtual void readValue(UnsignedLong & index, UnsignedLong & value);
	virtual void readValue(UnsignedLong & index, Bool & value);
	virtual void readValue(String & name, NumericalScalar & value);
	virtual void readValue(String & name, NumericalComplex & value);
	virtual void readValue(String & name, InterfaceObject & obj);
	virtual void readValue(String & name, PersistentObject & obj);
	virtual void readValue(String & name, String & st);
	virtual void readValue(String & name, UnsignedLong & value);
	virtual void readValue(String & name, Bool & value);
	virtual void readValue(InterfaceObject & obj, AttributeName attribute, const String & st);
	virtual void readValue(PersistentObject & obj, AttributeName attribute, const String & st);

      private:

	/** The file we read from/write to */
	FileName fileName_;

	/** Internal states */
	mutable StateStack state_;

	/** The internal document */
	XMLDoc document_;

	/** Methods to read DOM elements */
	virtual PersistentObject * readDOMElement() const;

      }; /* class XMLStorageManager */


    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_XMLSTORAGEMANAGER_HXX */
