//                                               -*- C -*-
/**
 * @file  WrapperCommon.h
 * @brief This file provides objects and functions commonly used by wrappers
 *
 * (C) Copyright 2005-2006 EDF
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * @author $LastChangedBy: dutka $
 * @date   $LastChangedDate: 2008-10-09 16:58:58 +0200 (jeu 09 oct 2008) $
 */

#ifndef OPENTURNS_WRAPPERCOMMON_H
#define OPENTURNS_WRAPPERCOMMON_H

#include <WrapperInterface.h>

#if defined(__cplusplus)
extern "C" {
#endif /* defined(__cplusplus) */

  typedef char const * wrapper_CHAR;
  typedef const wrapper_CHAR wrapper_CHARTAB[];

  /* The following declarations MUST be coherent with enumeration declarations in WrapperInterface.h */
  extern wrapper_CHARTAB WrapperListElementTypeAsString;
  extern wrapper_CHARTAB WrapperProvidedAsString;
  extern wrapper_CHARTAB WrapperConfigurationStateAsString;
  extern wrapper_CHARTAB WrapperConfigurationModeAsString;
  extern wrapper_CHARTAB WrapperDataTransferModeAsString;
  extern wrapper_CHARTAB WrapperErrorAsString;

  /** Return a string describing the error encountered */
  extern const char * getErrorAsString(enum WrapperErrorCode errorCode);

  /** Print out the user message passed as argument */
  extern void printUserMessage(const char * functionName, const char * message);

  /** Print out the message passed as argument */
  extern void printMessage(const char * functionName, const char * message);

  /** Print out a message telling that we enter a function */
  extern void printEntrance(const char * functionName);

  /** Print out a message telling that we leave a function */
  extern void printExit(const char * functionName);

  /** Print out a message telling which state we are working on */
  extern void printState(const char * functionName, void * p_state);

  /** Print out the content of the WrapperExchangedData structure */
  extern void printWrapperExchangedData(const char * functionName, const struct WrapperExchangedData * p_exchangedData);

  /* Print out the content of the WrapperExchangedData structure */
  extern void printWrapperInformation(const char * functionName, const struct WrapperInformation * p_info);

  /* Print out the content of a point structure */
  extern void printPoint(const char * functionName, const struct point * p_point);

  /* Print out the content of a sample structure */
  extern void printSample(const char * functionName, const struct sample * p_sample);

  /* Print out the content of a matrix structure */
  extern void printMatrix(const char * functionName, const struct matrix * p_matrix);

  /* Print out the content of a tensor structure */
  extern void printTensor(const char * functionName, const struct tensor * p_tensor);






  /* Get the number of variables of some type (in or out) in the data read from the description file */
  extern unsigned long getNumberOfVariables(const struct WrapperExchangedData * p_exchangedData, unsigned long type);

  /* Get the number of files of some type (in or out) in the data read from the description file */
  extern unsigned long getNumberOfFiles(const struct WrapperExchangedData * p_exchangedData, unsigned long type);

  /* Deep copy of a WrapperExchangedData structure (returns non-zero if error occured) */
  extern long copyWrapperExchangedData(struct WrapperExchangedData ** p_p_new_exchangedData, const struct WrapperExchangedData * p_exchangedData);

  /* Free a WrapperExchangedData structure */
  extern void freeWrapperExchangedData(struct WrapperExchangedData * p_exchangedData);

  /* Get the command line from the exchanged data structure */
  extern const char * getCommand(const struct WrapperExchangedData * p_exchangedData);

  /* Get the number of virtual CPUs from the exchanged data structure */
  extern unsigned long getNumberOfCPUs(const struct WrapperExchangedData * p_exchangedData);





  /* Create a temporary directory. This function internaly allocates the storage of the return value.
   * The symetric function deleteTemporaryDirectory should be used to avoid memory leak.
   */
  extern char * createTemporaryDirectory(const char * tempDirPrefix, const struct WrapperExchangedData * p_exchangedData);

  /* Delete a temporary directory if executionStatus is zero (ie, no error).
   * This function internaly frees the storage allocated by
   * createTemporaryDirectory, so be careful not to do it yourself. The content of the
   * directory is erased.
   */
  extern void deleteTemporaryDirectory(char * tempDir, long executionStatus);

  /* Change working directory to path */
  extern long changeDirectory(const char * path) DEPRECATED;

  /* Return the name of the current directory in a string. Note that this storage is
   * internaly allocated by the function, so users MUST free it (using free()).
   */
  extern char * getCurrentWorkingDirectory();

  /* Create the files listed in the exchanged data structure in the directory.
   * This action may result in variable substitution in the files according to the
   * regular expressions listed in the exchanged data structure. Returns non-zero if
   * some failure occured.
   */
  extern long createInputFiles(const char * directory, const struct WrapperExchangedData * p_exchangedData, const struct point * p_point);

  /* Read the content of the output listed in the exchanged data structure located in
   * the directory. Returns non-zero if some failure occured. For any output variable
   * not found in output file, the corresponding coordinate in p_point is set to NaN.
   */
  extern long readOutputFiles(const char * directory, const struct WrapperExchangedData * p_exchangedData, struct point * p_point);

  /* Make a command line from the template command and the variable list. The returned value is the new
   * command with variables substituted. This buffer is internaly allocated and MUST be deallocated
   * using free.
   */
  extern char * makeCommandFromTemplate(const char * command, const struct WrapperExchangedData * p_exchangedData, const struct point * p_point);

  /* Insulate a shell command, so that it can be run in a random temporary directory */
  extern char * insulateCommand(char * command, const char * temporaryDir);

  /* Run an insulated command based on the template read from the wrapper description file */
  extern long runInsulatedCommand(const char * temporaryDir, const struct WrapperExchangedData * p_exchangedData, const struct point * p_point);

#if defined(__cplusplus)
} /* end extern "C" */
#endif /* defined(__cplusplus) */ 

#endif /* OPENTURNS_WRAPPERCOMMON_H */
