//                                               -*- C++ -*-
/**
 *  @file  ResourceMap.cxx
 *  @brief ResourceMap defines top-most resourceMap strategies
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-10-29 18:54:45 +0100 (mer 29 oct 2008) $
 *  Id:      $Id: ResourceMap.cxx 990 2008-10-29 17:54:45Z dutka $
 */
#include <unistd.h>                 // for sysconf
#include "OSS.hxx"
#include "ResourceMap.hxx"
#include "Exception.hxx"
#include "Path.hxx"
#include "Log.hxx"
#include "XMLToolbox.hxx"

#ifndef R_EXECUTABLE_PATH
#error "R_EXECUTABLE_PATH not defined. Check configuration."
#endif

#ifndef XML_SUPPORTED
#error "XML support is mandatory. Check configuration."
#endif

namespace OpenTURNS
{

  namespace Base
  {

    namespace Common
    {

      static const char * ConfigurationFileName = "openturns.conf";
      static const char * RootElementName = "openturns-configuration";
      static const char * XMLTag_value = "value";
      static const char * XMLAttr_undefined = "undefined";

      Pointer<ResourceMap> ResourceMap::P_instance_;
      

      /* Since ResourceMap is a singleton, GetInstance gives access to the object */
      ResourceMap & ResourceMap::GetInstance()
      {
	if (P_instance_.isNull()) P_instance_ = new ResourceMap;
	return *P_instance_;
      }


      /* Default constructor */
      ResourceMap::ResourceMap()
	: std::map<String,String>()
      {
	loadDefaultConfiguration();
	loadConfigurationFile();
      }

      /* Method for retrieving information from the resource map */
      const String ResourceMap::get(KeyType key) const
      {
	std::map<String,String>::const_iterator it = std::map<String,String>::find(key);
	return (it != end()) ? it->second : "undefined";
      }

      /* Method for setting information into the resource map */
      void ResourceMap::set(KeyType key, MappedType value)
      {
	std::map<String,String>::operator[](key) = value;
      }

      /* Begin points to the first pair of the map */
      ResourceMap::iterator ResourceMap::begin()
      {
	return std::map<String,String>::begin();
      }

      ResourceMap::const_iterator ResourceMap::begin() const
      {
	return std::map<String,String>::begin();
      }


      /* End points past the last pair of the map */
      ResourceMap::iterator ResourceMap::end()
      {
	return std::map<String,String>::end();
      }

      ResourceMap::const_iterator ResourceMap::end() const
      {
	return std::map<String,String>::end();
      }

#if defined HAS_LIBXML2

      /* Update the ResourceMap with information from the configuration file */
      void ResourceMap::readConfigurationFile(const FileName & configurationFile)
      {
	Log::Info(OSS() << "Using configuration file " << configurationFile);

	// Open the configuration file and load it in a DOM document for later analysis
	XMLDoc document( configurationFile );

	// Check it is an OpenTURNS' one
	xmlNodePtr rootElt = xmlDocGetRootElement( document );
	if (rootElt == NULL) throw ConfigurationFileParsingException(HERE) << "Wrapper file has no root element" << configurationFile;
	if (xmlStrcmp( rootElt->name, REINTERPRET_CAST(const xmlChar *, RootElementName) ))
	  throw ConfigurationFileParsingException(HERE) << "Wrapper file " << configurationFile
						  << " has an invalid root element (" << rootElt->name << ")"
						  << " at line " << xmlGetLineNo(rootElt);


	// Walk through the DOM document to read the data
	if (rootElt && rootElt->children) {
	  for (xmlNodePtr current = rootElt->children; current; current = current->next) {
	    if (XML::IsElement(current)) {
	      String key = XML::ToString(current->name);
	      String value = XML::GetAttributeByName( current, XMLTag_value );
	      if (value.empty()) value = XMLAttr_undefined;
	      std::map<String,String>::operator[](key) = value;
	    }
	  }
	}

      }

#else
      /* Update the ResourceMap with information from the configuration file */
      void ResourceMap::readConfigurationFile(const FileName & configurationFile)
      {
	throw NotYetImplemented(HERE) << "ResourceMap only support libxml2";
      }

#endif

      /* Find the configuration file in specific path (see Path class for algorithm) */
      FileName ResourceMap::findConfigurationFile() const
      {
	Path::DirectoryList dirList = Path::GetConfigDirectoryList();
	return Path::FindFileByNameInDirectoryList(ConfigurationFileName, dirList);
      }

      /* Load the configuration file */
      void ResourceMap::loadConfigurationFile()
      {
	readConfigurationFile(findConfigurationFile());
      }

      /* Load the configuration defined at installation time */
      void ResourceMap::loadDefaultConfiguration()
      {
	std::map<String,String>::operator[]("R-executable-command")      = R_EXECUTABLE_PATH;
	std::map<String,String>::operator[]("temporary-directory")       = "/tmp";
	std::map<String,String>::operator[]("csv-file-separator")        = ";";
	std::map<String,String>::operator[]("parallel-threads")          = ( OSS() << sysconf(_SC_NPROCESSORS_CONF) );
	std::map<String,String>::operator[]("real-regexp-shortcut")      = "\\R";
	std::map<String,String>::operator[]("integer-regexp-shortcut")   = "\\I";
	std::map<String,String>::operator[]("separator-regexp-shortcut") = "\\S";
	std::map<String,String>::operator[]("real-regexp")               = "([+-]?[ \t]*([0-9]+(\\.[0-9]*)?|\\.[0-9]+)([Ee][+-]?[0-9]+)?)";
	std::map<String,String>::operator[]("integer-regexp")            = "([+-]?[ \t]*[0-9]+)";
	std::map<String,String>::operator[]("separator-regexp")          = "([ \t])";
	std::map<String,String>::operator[]("cache-max-size")            = "1024";
      }

      /* String converter */
      String ResourceMap::str() const
      {
	OSS oss;
        oss << "ResourceMap={";
	ResourceMap::const_iterator it;
	const char * separator = "\n  ";
	for(it=begin(); it!=end(); it++, separator=",\n  ")
	  oss << separator
	     << it->first << " => " << it->second;
	oss << "\n}";
	return oss;
      }



      /*
       * @fn std::ostream & operator <<(std::ostream & os, const ResourceMap & obj)
       * @brief Output stream converter
       * @param os A STL output stream resourceMap
       * @param obj The resourceMap read by \em os
       * @return A reference to \em os
       *
       * Operator << converts the ResourceMap object to an output stream
       * so it is easy to show the reason of the resourceMap.
       */
      std::ostream & operator <<(std::ostream & os, const ResourceMap & obj)
      {
        os << obj.str();
	return os;
      }





    } /* namespace Common */
  } /* namespace Base */
} /* namespace OpenTURNS */
