//                                               -*- C++ -*-
/**
 * @file		Study.cxx
 * @brief		Graphical representation of an OpenTURNS's Study
 *
 * @author		Romuald Conty
 * @date			2006-06-12
 *
 * @par Last change :
 *  $LastChangedBy: dutka $
 *  $LastChangedDate: 2007-11-08 17:13:40 +0100 (jeu 08 nov 2007) $
 *
 *  (C) Copyright 2005-2007 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

// Header
#include "Study.hxx"

// OT::UI::GUI
#include "QOTFactory.hxx"

//Qt
#include <qfile.h>

namespace OpenTURNS
{
	namespace UI
	{
		namespace GUI
		{
			/** Default constructor */
			Study::Study ( const char *name ) : QOTObject ( NULL, name )
			{
				filename_ = "";
				isModified_ = true;

				stepA_ = new StepA ( this );
				stepB_ = new StepB ( this );

				stepC_ = new Step ( this, "Step C" );
				stepC_->setName ( tr ( "Step C" ) );
				stepC_->setDescription ( tr ( "Results" ) );

				stepC2_ = new StepC2 ( stepC_, this );
				stepC3_ = new StepC3 ( stepC_, this );

				setRenameEnabled ( true );
			}

			/** Destructor */
			Study::~Study()
			{
				delete stepA_;
				delete stepB_;
				delete stepC_;
			}

			/**
			 * @fn OpenTURNS::Bool Study::save()
			 * @return return true if saving have been done otherwise return false.
			 */
			bool Study::save()
			{
				// Save..
				QFile f ( getFilename() );
				if ( !f.open ( IO_WriteOnly ) )
				{
					// 				statusBar()->message( tr("Could not write to %1").arg( getFilename() ), 2000 );
					return false;
				}
				QDomDocument doc ( "QOpenTURNS" );
				doc.appendChild ( this->toXmlNode ( doc ) );

				QTextStream t ( &f );
				t << doc.toString();

				f.close();

				setIsModified ( false );
				return true;
			}

			/**
			 * @fn OpenTURNS::Bool Study::load(QString filename)
			 *
			 * @return return true if loading have been done otherwise return false.
			 */
			/**
			 * @todo use exception to handle loading errors
			 */
			bool Study::load ( const QString& filename )
			{
				// Load..
				setIsModified ( false );
				QFile f ( filename );
				if ( !f.open ( IO_ReadOnly ) )
					return false;

				QOTFactory * factory = new QOTFactory ( this );
				bool res = factory->loadFromXml ( &f );
				return res;
			}

			/**
			 * @fn OpenTURNS::Bool Study::isModified()
			 * @return return true if the study have been modified since last save.
			 */
			bool Study::isModified()
			{
				return isModified_;
			}

			/** Study isModified_ accessor */
			void Study::setIsModified ( bool newState )
			{
				if ( isModified_ != newState )
				{
					isModified_ = newState;
					emit modifiedStateChanged ( newState );
				}
			}

			/**
			 * @fn void Study::setFilename(const QString& filename)
			 * @brief Study filename accessor
			 * @param filename a path to file to use.
			 */
			void Study::setFilename ( const QString& filename )
			{
				filename_ = filename;
			}

			/**
			 * @fn QString Study::getFilename() const
			 * @brief Study filename accessor
			 * @returns return true if the study have been modified since last save.
			 */
			QString Study::getFilename() const
			{
				return filename_;
			}

			QString Study::getFullName() const
			{
				return getName();
			}

			/**
			 * @fn StepA* Study::getStepA()
			 * @brief Study StepA accessor
			 * @return the study's StepA.
			 */
			StepA* Study::getStepA()
			{
				return stepA_;
			}

			/**
			 * @fn StepB* Study::getStepB()
			 * @brief Study StepB accessor
			 * @return the study's StepB.
			 */
			StepB* Study::getStepB()
			{
				return stepB_;
			}

			/**
			 * @fn StepC2* Study::getStepC2()
			 * @brief Study StepC2 accessor
			 * @return the study's StepC.
			 */
			StepC2* Study::getStepC2()
			{
				return stepC2_;
			}

			/**
			 * @fn StepC3* Study::getStepC3()
			 * @brief Study StepC3 accessor
			 * @return the study's StepC.
			 */
			StepC3* Study::getStepC3()
			{
				return stepC3_;
			}
		} /* namespace GUI */
	} /* namespace UI */
} /* namespace OpenTURNS */

